/*******************************************************************************
 * Copyright 2016-2019 by SW Group, Chengdu Haiguang IC Design Co., Ltd.
 * All right reserved. See COPYRIGHT for detailed Information.
 *
 * @file        ptx_driver.h
 * @brief       The header file of the ptx driver.
 *
 * @author      Yu Anxiong<yuanxiong@hygon.cn>
 * @date        2024/03/15
 * @history     1.
 ******************************************************************************/

#ifndef __INC_DRIVER_PTX_DRIVER_H__
#define __INC_DRIVER_PTX_DRIVER_H__

#include <stdint.h>

#ifndef __cplusplus
#include <stdbool.h>  /* bool */
#endif /* __cplusplus */

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @brief Opaque handle. 
 */
typedef int64_t PtxDriver;

/**
 * @brief   Set the log Level, if the log level is not supported, set it  
 *          to the default value.
 * 
 * @param log_level   The log level, optional value: warning, info, debug.
 *                    Default is info.
 * 
 * @return   return void.
 */
void PtxSetLogLevel(const char* log_level);

/**
 * @brief   Create a ptx driver handle, which is required for subsequent 
 *          operations.
 * 
 * @param cfg_dir     If not NULL, generate control flow graph file in the 
 *                    specified directory.
 * @param dfg_dir     If not NULL, generate data flow graph file in the 
 *                    specified directory.
 * 
 * @return   return ptx driver handle if success, 0 otherwise.
 */
PtxDriver PtxCreateDriver(const char* cfg_dir, const char* dfg_dir);

/**
 * @brief  Destroy the specified ptx driver handle.
 * 
 * @param ptx_driver  Already created ptx driver handle.
 * 
 * @return   return true if success, false otherwise.
 */
bool PtxDestroyDriver(PtxDriver ptx_driver);

/**
 * @brief   Translate input file to target language.
 * 
 * @param ptx_driver  Already created ptx driver handle.
 * @param input_file  Source file of ptx code.
 * @param target_arch The target architecture that needs to be translated.
 * @return   return true if success, false otherwise.
 */
bool PtxTranslateFile(PtxDriver ptx_driver, 
                      const char* input_file, 
                      const char* target_arch);

/**
 * @brief   Translate input text to target language.
 * 
 * @param ptx_driver  Already created ptx driver handle.
 * @param input_text  Source text of ptx code.
 * @param target_arch The target architecture that needs to be translated.
 * @return   return true if success, false otherwise.
 */
bool PtxTranslateText(PtxDriver ptx_driver, 
                      const char* input_text, 
                      const char* target_arch);

/**
 * @brief   Get the translated text length.
 * 
 * @param ptx_driver  Already created ptx driver handle.
 * @return   return translated text length if success, -1 otherwise.
 */
int  PtxGetTranslatedTextSize(PtxDriver ptx_driver);

/**
 * @brief   Get the translated text.
 * 
 * @param ptx_driver  Already created ptx driver handle.
 * @param buffer      Store the translated text.
 * @param size        The size of the text to be copied.
 * @return   return true if success, false otherwise.
 */
bool PtxGetTranslatedText(PtxDriver ptx_driver, char* buffer, int size);

#ifdef __cplusplus
}   //extern "C"
#endif

#endif // __INC_DRIVER_PTX_DRIVER_H__
