/******************************************************************************
 * Copyright 2016-2019 by SW Group, Chengdu Haiguang IC Design Co., Ltd.
 * All right reserved. See COPYRIGHT for detailed Information.
 *
 * @file        hsa_ext_log.h
 * @brief       HSA log API.
 *
 * @author      Peng Xinghong<pengxinghong@hygon.cn>
 * @date        2024/04/11
 * @history     1.
 *****************************************************************************/

#ifndef HSA_EXT_LOG_H
#define HSA_EXT_LOG_H

#include "hylog.h"

#ifdef __cplusplus
extern "C" {
#endif /*__cplusplus*/ 
/**
 * @brief HSA log level.
 */
typedef enum {
  /**
   * Bad stuff that shouldn't happen. The system broke its contract to
   * users in some way or some major assumption was violated.
   */
  HSA_EXT_LOG_ERROR = 1,
  /**
   * Messages at the WARNING level indicate that, although something went
   * wrong or something unexpected happened, it was transient and
   * recoverable.
   */
  HSA_EXT_LOG_WARNING,
  /**
   * Somewhere in between WARNING and DEBUG...
   */
  HSA_EXT_LOG_INFO,
  /**
   * Messages at the DEBUG level don't necessarily indicate that anything
   * went wrong, but they could be useful in diagnosing problems.
   */
  HSA_EXT_LOG_DEBUG
}hsa_ext_log_level_t;

/**
 * Returns the current minimum log severity level enforced by hylog
 */
HyLogLevel hsa_ext_log_get_level();

/**
 * Sets the minimum logging severity level in the system. All log statements
 * of a lower log severity will be dropped completely.
 *
 * \param logLevel
 *      New Log level to set
 */
void hsa_ext_log_set_level(HyLogLevel logLevel);

/**
 * Waits until all pending log statements are persisted to disk. Note that if
 * there is another logging thread continually adding new pending log
 * statements, this function may not return until all threads stop logging and
 * all the new log statements are also persisted.
 */
void hsa_ext_log_sync();

#define hsa_ext_log(level, fmt, ...)                          \
  do {                                                        \
      HY_LOG((HyLogLevel)(level), fmt, ##__VA_ARGS__);        \
  } while (false)

#ifdef __cplusplus
}  // end extern "C" block
#endif /*__cplusplus*/ 

#endif
