/******************************************************************************
 * Copyright 2016-2019 by SW Group, Chengdu Haiguang IC Design Co., Ltd.
 * All right reserved. See COPYRIGHT for detailed Information.
 *
 * @file        main.cpp
 * @brief       The hdms is hygon device management service.
 *
 * @author      Wang Yan<wangwy@hygon.cn>
 * @date        2022/10/18
 * @history     1.
 *****************************************************************************/
#include <unistd.h>
#include <cstring>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include "arguments.h"
#include "utils.h"
#include "dmi.h"
#include "logger.h"
#include "version.h"

#define PrintVersion(is_print)                                           \
    do {                                                                 \
        if (is_print) {                                                  \
            PrintInfo("Version %d.%d.%d (%s %s)\n", MAJOR_VERSION,       \
                      MINOR_VERSION, PATCH_VERSION, __DATE__, __TIME__); \
            exit(0);                                                     \
        }                                                                \
    } while (false);

#define PrintHelp(is_print, help_info)                                   \
    do {                                                                 \
        if (is_print) {                                                  \
            PrintInfo(help_info);                                        \
            exit(0);                                                     \
        }                                                                \
    } while (false);

#define CheckOption(opt0, opt_str0, opt1, opt_str1)                      \
    do {                                                                 \
        if ((opt0) && (opt1)) {                                          \
            PrintError(                                                  \
                "The %s and %s options must be specified "               \
                "at the same time\n",                                    \
                opt_str0, opt_str1);                                     \
            return false;                                                \
        }                                                                \
    } while (false);

#define DmiCheck(status, prefix_msg)                                     \
    do {                                                                 \
        dmiStatus local_status = status;                                 \
        if (local_status != DMI_STATUS_SUCCESS) {                        \
            const char* error_msg = nullptr;                             \
            dmiGetStatusString(local_status, &error_msg);                \
            PrintError("%s%s\n", prefix_msg, error_msg);                 \
            return false;                                                \
        }                                                                \
    } while (false);

static bool device_is_start = false;
int virtual_device_id       = -1;

/* handle signal */
void INThandler(int sig) {
    if (virtual_device_id >= 0 && device_is_start) {
        dmiStatus status = dmiStopVDevice(virtual_device_id);
        if (status != DMI_STATUS_SUCCESS) {
            const char* error_msg = nullptr;
            dmiGetStatusString(status, &error_msg);
            PrintError("Failed to stop vdevice: %s\n", error_msg);
        }
        device_is_start = false;
    }
}

bool InitLinuxSignalsHandler(void) {
    // ctrl + c
    if (signal(SIGINT, INThandler) == SIG_ERR) {
        PrintError("Failed to add SIGINT: %s", strerror(errno));
        return false;
    }

    return true;
}

void doSignalHandler(int status) {
    bool coredump = (status >> 7) & 0x1;
    std::string str;

    switch (WTERMSIG(status)) {
        case SIGILL: str += "Illegal instruction"; break;
        case SIGABRT: str += "Aborted"; break;
        case SIGSEGV: str += "Segmentation fault"; break;
        default: str += StrFormat("Exit signal: %d", WTERMSIG(status)); break;
    }

    if (coredump) { str += " (core dumped)"; }

    PrintError("%s\n", str.c_str());
}

static bool DoMain(int argc, char** argv) {
    utils::Arguments args(argc, argv);
    args.SetDescription("The hdms is hygon dcu device management service.");

    std::string log_level("info"), log_file("");
    bool print_version = false, print_help = false;
    std::string procs;

    args.SetGroupTitle("Options:");
    args.Parse("-log-level", "--log-level", log_level, "<level>",
               "Set the log level: info(default), debug, trace.");
    args.Parse("-log-file", "--log-file", log_file, "<file>",
               "Set the file used to write the log.");
    args.Parse("", "--version", print_version,
               "Print version information and exit.");
    args.Parse("-h", "--help", print_help, "Print available options and exit.");

    args.SetGroupTitle("Virtualization options:");
    args.Parse("-p", "--process", procs, "<process>",
               "Specifies the running process.");
    args.Parse("-v", "--vdevice", virtual_device_id, "<number>",
               "Specifies the running vdevs.");

    // Try to set the log first, because of later dependencies
    utils::SetLogLevel(log_level);
    utils::SetLogFile(log_file);
    std::string errmsg = args.GetErrorMessage();
    if (!errmsg.empty()) {
        PrintError(errmsg + "\n");
        return false;
    }

    if (argc < 2) { print_help = true; }

    PrintVersion(print_version);
    PrintHelp(print_help, args.GetHelpInfo());

    InitLinuxSignalsHandler();

    const char* vdev_opt  = "-v/--vdevice";
    const char* procs_opt = "-p/--process";

    if (virtual_device_id >= 0) {
        CheckOption(true, vdev_opt, procs.empty(), procs_opt);
        DmiCheck(dmiStartVDevice(virtual_device_id),
                 "Failed to start vdevice: ");
        device_is_start = true;
    }

    pid_t pid = -1;
    if (!procs.empty()) {
        PrintDebug("Process: " + procs + "\n");
        std::stringstream s(procs);
        std::vector<std::string> vector_args;
        std::string value;
        while (s >> value) {
            vector_args.push_back(value);
            PrintDebug("push: " + value + " into vector args.\n");
        }

        int n      = vector_args.size();
        char** cmd = new char*[n + 1];
        for (int i = 0; i < n; ++i) {
            cmd[i] = const_cast<char*>(vector_args.at(i).c_str());
        }
        cmd[n] = nullptr;

        pid = vfork();
        if (0 > pid) { PrintError("vfork failed.\n") }

        if (0 == pid) {
            // Execute the program
            if (execvp(cmd[0], cmd) == -1) {
                perror("Execute the program failed");
                exit(-1);
            }
        } else {
            // Parent - wait for program to complete
            int status;
            pid_t wait_pid = wait(&status);
            if (wait_pid == -1) {
                perror("wait error :");
            } else if (WIFSIGNALED(status)) {
                doSignalHandler(status);
            }

            delete[] cmd;
        }
    }

    if (virtual_device_id >= 0 && device_is_start) {
        DmiCheck(dmiStopVDevice(virtual_device_id), "Failed to stop vdevice: ");
        device_is_start = false;
    }

    return true;
}

int main(int argc, char** argv) {
    try {
        return DoMain(argc, argv) ? EXIT_SUCCESS : EXIT_FAILURE;
    } catch (const std::exception& e) {
        PrintError("%s\n", e.what());
        return EXIT_FAILURE;
    } catch (...) {
        PrintError("Unhandled exception\n");
        return EXIT_FAILURE;
    }
}
