/* Copyright (c) 2016-2020 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR(S) DISCLAIM ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL AUTHORS BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef HYLOG_RAW_DATA_H
#define HYLOG_RAW_DATA_H

#include <cstdint>

/**
 * The Log namespace contains a collection of data structures and functions
 * to manage the metadata in the various buffers of the hylog system.
 * Here the term "compressed log" shall be used to refer to the runtime output
 * of the hylog system and the "uncompressed log" refers to the entries within
 * the StagingBuffers.
 *
 * The rough interaction between the various components of hylog involving
 * buffers is detailed in the diagrams below:
 *
 * At runtime:
 * (Logging Code) ==> StagingBuffer ==> Encoder ==> (Compressed Log File)
 *
 * At post execution/decompression
 * (Compressed Log File) ==> Decoder ===> (Human-Readable Output)
 *
 * The format of the StagingBuffer looks something like this:
 * **************************
 * * UncompressedLogEntry   *
 * **************************
 * * Uncompressed Arguments *
 * **************************
 * *        .....           *
 *
 * This file controls format of UncompressedLogEntry, but everything else
 * in the StagingBuffer (i.e. UncompressedArguments) is opaque. How it's laid
 * out and compressed is controlled by the users of the StagingBuffer
 * (i.e. generated code for Preprocessor hylog or store.h for C++17
 * hylog).
 *
 * The format of the Compressed Log looks something like this
 * *****************
 * *  Checkpoint   *
 * * ------------- *
 * *  Dictionary   *
 * * ------------- *
 * * Buffer Extent *
 * *   (LogMsgs)   *
 * * --------------*
 * *     ....      *
 *
 * The overall layout of the Compressed Log is controlled by the Encoder and
 * Decoder classes below, but how the LogMsgs are encoded is again controlled
 * by outside code.
 */
namespace HyLogInternal {

/**
 * Describes the type of parameter that would be passed into a printf-like
 * function.
 *
 * These types are optimized to store enough information to determine
 * (a) whether a 'const char*' parameter indicates string (%s) or not (%p)
 * (b) if a string parameter (%s) needs to be truncated due to precision
 * (c) whether a parameter is a dynamic precision/width specifier
 */
enum ParamType : int32_t {
    // Indicates that there is a problem with the parameter
    INVALID = -6,

    // Indicates a dynamic width (i.e. the '*' in  %*.d)
    DYNAMIC_WIDTH = -5,

    // Indicates dynamic precision (i.e. the '*' in %.*d)
    DYNAMIC_PRECISION = -4,

    // Indicates that the parameter is not a string type (i.e. %d, %lf)
    NON_STRING = -3,

    // Indicates the parameter is a string and has a dynamic precision
    // (i.e. '%.*s' )
    STRING_WITH_DYNAMIC_PRECISION = -2,

    // Indicates a string with no precision specified (i.e. '%s' )
    STRING_WITH_NO_PRECISION = -1,

    // All non-negative values indicate a string with a precision equal to its
    // enum value casted as an int32_t
    STRING = 0
};

// Default, uninitialized value for log identifiers associated with log
// invocation sites.
static constexpr int UNASSIGNED_LOGID = -1;

/**
 * Stores the static log information associated with a log invocation site
 * (i.e. filename/line/fmtString combination).
 */
struct StaticLogInfo {

    // Function signature of the compression function used in the
    // non-preprocessor version of hylog
    typedef void (*CompressionFn)(int, const ParamType*, char**, char**);

    // Constructor
    constexpr StaticLogInfo(CompressionFn compress,
                      const char* filename,
                      const uint32_t lineNum,
                      const uint8_t severity,
                      const char* fmtString,
                      const int numParams,
                      const int numNibbles,
                      const ParamType* paramTypes)
            : compressionFunction(compress)
            , filename(filename)
            , lineNum(lineNum)
            , severity(severity)
            , formatString(fmtString)
            , numParams(numParams)
            , numNibbles(numNibbles)
            , paramTypes(paramTypes)
    { }

    // Stores the compression function to be used on the log's dynamic arguments
    CompressionFn compressionFunction;

    // File where the log invocation is invoked
    const char *filename;

    // Line number in the file for the invocation
    const uint32_t lineNum;

    // Log level severity associated with the log invocation
    const uint8_t severity;

    // printf format string associated with the log invocation
    const char *formatString;

    // Number of arguments required for the log invocation
    const int numParams;

    // Number of nibbles needed to compress all the non-string log arguments
    const int numNibbles;

    // Mapping of parameter index (i.e. order in which it appears in the
    // argument list starting at 0) to parameter type as inferred from the
    // printf log message invocation
    const ParamType* paramTypes;
};

namespace Log {
    /**
     * Marks the beginning of a log entry within the StagingBuffer waiting
     * for compression. Every instance of this header in the StagingBuffer
     * corresponds to a user invocation of the log function in the hylog
     * system and thus every field is uncompressed to lower the compute time
     * for that invocation.
     */
    struct UncompressedEntry {
        // Uniquely identifies a log message by its format string and file
        // location, assigned at compile time by the preprocessor.
        uint32_t fmtId;

        // Number of bytes for this header and the various uncompressed
        // log arguments after it
        uint32_t entrySize;

        // Stores the rdtsc() value at the time of the log function invocation
        uint64_t timestamp;

        // After this header are the uncompressed arguments required by
        // the original format string
        char argData[0];
    };
}; /* namespace Log */
}; /* namespace HyLogInternal */

#endif /* HYLOG_RAW_DATA_H */
