/* Copyright (c) 2016-2018 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR(S) DISCLAIM ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL AUTHORS BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
 
#ifndef HYLOG_H
#define HYLOG_H

/**
 * This header serves as the application and generated code interface into
 * the hylog Runtime system. This should be included where-ever the HY_LOG
 * macro is used.
 */

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */
/**
 * The levels of verbosity for messages logged with #HY_LOG.
 */
enum HyLogLevel {
    // Keep this in sync with logLevelNames defined inside Log.cc.
    HY_LOG_SILENT = 0,

    /**
     * Bad stuff that shouldn't happen. The system broke its contract to
     * users in some way or some major assumption was violated.
     */
    HY_LOG_ERROR,

    /**
     * Messages at the WARNING level indicate that, although something went
     * wrong or something unexpected happened, it was transient and
     * recoverable.
     */
    HY_LOG_WARNING,

    /**
     * Somewhere in between WARNING and DEBUG...
     */
    HY_LOG_NOTICE,

    /**
     * Messages at the DEBUG level don't necessarily indicate that anything
     * went wrong, but they could be useful in diagnosing problems.
     */
    HY_LOG_DEBUG,

    // must be the last element in the enum
    HY_LOG_LEVEL_NUM
};


// User API
/**
 * Returns the current minimum log severity level enforced by hylog
 */
HyLogLevel hsa_ext_log_get_level();

/**
 * Sets the minimum logging severity level in the system. All log statements
 * of a lower log severity will be dropped completely.
 *
 * \param logLevel
 *      New Log level to set
 */
void hsa_ext_log_set_level(HyLogLevel logLevel);

/**
 * Waits until all pending log statements are persisted to disk. Note that if
 * there is another logging thread continually adding new pending log
 * statements, this function may not return until all threads stop logging and
 * all the new log statements are also persisted.
 */
void hsa_ext_log_sync();
#ifdef __cplusplus
}
#endif  /* __cplusplus */

#include "portability.h"
#include "raw_data.h"
#include "packer.h"
#include "store.h"
/**
 * HY_LOG macro used for logging.
 *
 * \param severity
 *      The log level of the log invocation (must be constant)
 * \param format
 *      printf-like format string (must be literal)
 * \param ...UNASSIGNED_LOGID
 *      Log arguments associated with the printf-like string.
 */
#define HY_LOG(severity, format, ...) do { \
    constexpr int numNibbles = HyLogInternal::getNumNibblesNeeded(format); \
    constexpr int nParams = HyLogInternal::countFmtParams(format); \
    constexpr int relPathOff = HyLogInternal::getRelPathOff(__PROJECT_DIR__); \
    \
    /*** Very Important*** These must be 'static' so that we can save pointers
     * to these variables and have them persist beyond the invocation.
     * The static logId is used to forever associate this local scope (tied
     * to an expansion of #HY_LOG) with an id and the paramTypes array is
     * used by the compression function, which is invoked in another thread
     * at a much later time. */ \
    static constexpr std::array<HyLogInternal::ParamType, nParams> paramTypes = \
                                HyLogInternal::analyzeFormatString<nParams>(format); \
    static int logId = HyLogInternal::UNASSIGNED_LOGID; \
    \
    /* Triggers the GNU printf checker by passing it into a no-op function.
     * Trick: This call is surrounded by an if false so that the VA_ARGS don't
     * evaluate for cases like '++i'.*/ \
    if (false) { HyLogInternal::checkFormat(format, ##__VA_ARGS__); } /*NOLINT(cppcoreguidelines-pro-type-vararg, hicpp-vararg)*/\
    \
    if (hsa_ext_log_check(&__FILE__[relPathOff], __LINE__, severity, format, ##__VA_ARGS__)) { \
        HyLogInternal::log(logId, &__FILE__[relPathOff], __LINE__, severity, format, \
                             numNibbles, paramTypes, ##__VA_ARGS__); \
    } \
} while(0)

#endif // HYLOG_H
