#!/usr/bin/env bash
# Set bash to 'debug' mode, it will exit on :
# -e 'error', -u 'undefined variable', -o ... 'error in pipeline', -x 'print commands',
set -e
set -u
set -o pipefail

N=6
log() {
    local fname=${BASH_SOURCE[1]##*/}
    echo -e "$(date '+%Y-%m-%dT%H:%M:%S') (${fname}:${BASH_LINENO[0]}:${FUNCNAME[1]}) $*"
}
help_message=$(cat << EOF
Usage: $0

Options:
    --remove_archive (bool): true or false
      With remove_archive=True, the archives will be removed after being successfully downloaded and un-tarred.
EOF
)
SECONDS=0

# Data preparation related
data_url=www.openslr.org/resources/33
remove_archive=false
download_opt=

log "$0 $*"


. ./utils/parse_options.sh

. ./db.sh
. ./path.sh
. ./cmd.sh


if [ $# -gt 1 ]; then
  log "${help_message}"
  exit 2
fi

if "$remove_archive"; then
  download_opt="--remove-archive"
fi

if [ -z "${AISHELL}" ]; then
  log "Error: \$AISHELL is not set in db.sh."
  exit 2
fi


log "Download data to ${AISHELL}"
if [ ! -d "${AISHELL}" ]; then
    mkdir -p "${AISHELL}"
fi
# To absolute path
AISHELL=$(cd ${AISHELL}; pwd)

echo local/download_and_untar.sh ${download_opt} "${AISHELL}" "${data_url}" data_aishell
local/download_and_untar.sh ${download_opt} "${AISHELL}" "${data_url}" data_aishell
echo local/download_and_untar.sh ${download_opt} "${AISHELL}" "${data_url}" resource_aishell
local/download_and_untar.sh ${download_opt} "${AISHELL}" "${data_url}" resource_aishell

aishell_audio_dir=${AISHELL}/data_aishell/wav
aishell_text=${AISHELL}/data_aishell/transcript/aishell_transcript_v0.8.txt

log "Data Preparation"
train_dir=data${N}/local/train
dev_dir=data${N}/local/dev
test_dir=data${N}/local/test
tmp_dir=data${N}/local/tmp

mkdir -p $train_dir
mkdir -p $dev_dir
mkdir -p $test_dir
mkdir -p $tmp_dir

# find wav audio file for train, dev and test resp.
find $aishell_audio_dir -iname "*.wav" > $tmp_dir/wav.flist
n=$(wc -l < $tmp_dir/wav.flist)
[ $n -ne 141925 ] && \
  log Warning: expected 141925 data data files, found $n

grep -i "wav/train" $tmp_dir/wav.flist > $train_dir/wav.flist || exit 1;
grep -i "wav/dev" $tmp_dir/wav.flist > $dev_dir/wav.flist || exit 1;
grep -i "wav/test" $tmp_dir/wav.flist > $test_dir/wav.flist || exit 1;
echo gaoruiqi
rm -r $tmp_dir

echo gaoruiqi
# Transcriptions preparation
for dir in $train_dir $dev_dir $test_dir; do
  log Preparing $dir transcriptions
  sed -e 's/\.wav//' $dir/wav.flist | awk -F '/' '{print $NF}' > $dir/utt.list
  sed -e 's/\.wav//' $dir/wav.flist | awk -F '/' '{i=NF-1;printf("%s %s\n",$NF,$i)}' > $dir/utt2spk_all
  paste -d' ' $dir/utt.list $dir/wav.flist > $dir/wav.scp_all
  utils/filter_scp.pl -f 1 $dir/utt.list $aishell_text > $dir/transcripts.txt
  awk '{print $1}' $dir/transcripts.txt > $dir/utt.list
  utils/filter_scp.pl -f 1 $dir/utt.list $dir/utt2spk_all | sort -u > $dir/utt2spk
  utils/filter_scp.pl -f 1 $dir/utt.list $dir/wav.scp_all | sort -u > $dir/wav.scp
  sort -u $dir/transcripts.txt > $dir/text
  utils/utt2spk_to_spk2utt.pl $dir/utt2spk > $dir/spk2utt
done

mkdir -p data${N}/train data${N}/dev data${N}/test

for f in spk2utt utt2spk wav.scp text; do
  cp $train_dir/$f data${N}/train/$f || exit 1;
  cp $dev_dir/$f data${N}/dev/$f || exit 1;
  cp $test_dir/$f data${N}/test/$f || exit 1;
done

# remove space in text
for x in train dev test; do
  cp data${N}/${x}/text data${N}/${x}/text.org
  paste -d " " <(cut -f 1 -d" " data${N}/${x}/text.org) <(cut -f 2- -d" " data${N}/${x}/text.org | tr -d " ") \
      > data${N}/${x}/text
  echo gaoruiqi
  rm data${N}/${x}/text.org
  echo gaoruiqi
done

log "Successfully finished. [elapsed=${SECONDS}s]"
