/*
*
# Copyright (c) 2024, HCUOpt CORPORATION.  All rights reserved.
*
*/
package podman

import (
	"dtk-container-toolkit/internal/logger"
	"fmt"
	"gopkg.in/ini.v1"
	"os"
)

type builder struct {
	logger logger.Interface
	path   string
}

type Option func(*builder)

func WithLogger(logger logger.Interface) Option {
	return func(b *builder) {
		b.logger = logger
	}
}

func WithPath(path string) Option {
	return func(b *builder) {
		b.path = path
	}
}

func (b *builder) build() (*Config, error) {
	if b.path == "" {
		empty := ini.Empty()
		return &Config{File: empty}, nil
	}

	return b.loadConfig(b.path)
}

func (b *builder) loadConfig(config string) (*Config, error) {
	info, err := os.Stat(config)
	if os.IsExist(err) && info.IsDir() {
		return nil, fmt.Errorf("config file is a directory")
	}

	cfg := ini.Empty()

	if os.IsNotExist(err) {
		b.logger.Infof("Config file does not exist, using empty config")
		return &Config{File: cfg}, nil
	}

	b.logger.Infof("Loading config from %v", config)
        cfg, err = ini.LoadSources(ini.LoadOptions{
		IgnoreInlineComment:     false,
		AllowBooleanKeys:        true,
		PreserveSurroundedQuote: true, // 保留引号
	}, config)
	if err != nil {
		return nil, fmt.Errorf("unable to read config: %v", err)
	}
	return &Config{File: cfg}, nil
}

