/**
# Copyright (c) 2024, HCUOpt CORPORATION.  All rights reserved.
**/

package main

import (
	"dtk-container-toolkit/cmd/dtk-ctk/cdi"
	"dtk-container-toolkit/cmd/dtk-ctk/config"
	"dtk-container-toolkit/cmd/dtk-ctk/hook"
	"dtk-container-toolkit/cmd/dtk-ctk/runtime"
	"dtk-container-toolkit/internal/info"
	"os"

	"github.com/sirupsen/logrus"
	cli "github.com/urfave/cli/v2"
)

// options defines the options that can be set for the CLI through config files,
// environment variables, or command line flags
type options struct {
	// Debug indicates whether the CLI is started in "debug" mode
	Debug bool
	// Quiet indicates whether the CLI is started in "quiet" mode
	Quiet bool
}

func main() {
	logger := logrus.New()

	// Create a options struct to hold the parsed environment variables or command line flags
	opts := options{}

	// Create the top-level CLI
	c := cli.NewApp()
	c.Name = "C-3000 DTK Container Toolkit CLI"
	c.UseShortOptionHandling = true
	c.EnableBashCompletion = true
	c.Usage = "Tools to configure the C-3000 Container Toolkit"
	c.Version = info.GetVersionString()

	// Setup the flags for this command
	c.Flags = []cli.Flag{
		&cli.BoolFlag{
			Name:        "debug",
			Aliases:     []string{"d"},
			Usage:       "Enable debug-level logging",
			Destination: &opts.Debug,
			EnvVars:     []string{"DTK_CTK_DEBUG"},
		},
		&cli.BoolFlag{
			Name:        "quiet",
			Usage:       "Suppress all output except for errors; overrides --debug",
			Destination: &opts.Quiet,
			EnvVars:     []string{"DTK_CTK_QUIET"},
		},
	}

	// Set log-level for all subcommands
	c.Before = func(c *cli.Context) error {
		logLevel := logrus.InfoLevel
		if opts.Debug {
			logLevel = logrus.DebugLevel
		}
		if opts.Quiet {
			logLevel = logrus.ErrorLevel
		}
		logger.SetLevel(logLevel)
		return nil
	}

	// Define the subcommands
	c.Commands = []*cli.Command{
		runtime.NewCommand(logger),
		config.NewCommand(logger),
		hook.NewCommand(logger),
		cdi.NewCommand(logger),
	}

	// Run the CLI
	err := c.Run(os.Args)
	if err != nil {
		logger.Errorf("%v", err)
		os.Exit(1)
	}
}
