/**
# Copyright (c) 2024, HCUOpt CORPORATION.  All rights reserved.
**/

package crio

import (
	"dtk-container-toolkit/pkg/config/engine"
	"fmt"

	"github.com/pelletier/go-toml"
)

// Config represents the cri-o config
type Config toml.Tree

var _ engine.Interface = (*Config)(nil)

// New creates a cri-o config with the specified options
func New(opts ...Option) (engine.Interface, error) {
	b := &builder{}
	for _, opt := range opts {
		opt(b)
	}

	return b.build()
}

// AddRuntime adds a new runtime to the crio config
func (c *Config) AddRuntime(name string, path string, setAsDefault bool, _ ...map[string]interface{}) error {
	if c == nil {
		return fmt.Errorf("config is nil")
	}

	config := (toml.Tree)(*c)

	if runc, ok := config.Get("crio.runtime.runtimes.runc").(*toml.Tree); ok {
		runc, _ = toml.Load(runc.String())
		config.SetPath([]string{"crio", "runtime", "runtimes", name}, runc)
	}

	config.SetPath([]string{"crio", "runtime", "runtimes", name, "runtime_path"}, path)
	config.SetPath([]string{"crio", "runtime", "runtimes", name, "runtime_type"}, "oci")

	if setAsDefault {
		config.SetPath([]string{"crio", "runtime", "default_runtime"}, name)
	}

	*c = (Config)(config)
	return nil
}

// DefaultRuntime returns the default runtime for the cri-o config
func (c Config) DefaultRuntime() string {
	config := (toml.Tree)(c)
	if runtime, ok := config.GetPath([]string{"crio", "runtime", "default_runtime"}).(string); ok {
		return runtime
	}
	return ""
}

// RemoveRuntime removes a runtime from the cri-o config
func (c *Config) RemoveRuntime(name string) error {
	if c == nil {
		return nil
	}

	config := (toml.Tree)(*c)
	if runtime, ok := config.GetPath([]string{"crio", "runtime", "default_runtime"}).(string); ok {
		if runtime == name {
			config.DeletePath([]string{"crio", "runtime", "default_runtime"})
		}
	}

	runtimeClassPath := []string{"crio", "runtime", "runtimes", name}
	config.DeletePath(runtimeClassPath)
	for i := 0; i < len(runtimeClassPath); i++ {
		remainingPath := runtimeClassPath[:len(runtimeClassPath)-i]
		if entry, ok := config.GetPath(remainingPath).(*toml.Tree); ok {
			if len(entry.Keys()) != 0 {
				break
			}
			config.DeletePath(remainingPath)
		}
	}

	*c = (Config)(config)
	return nil
}

// Set sets the specified cri-o option.
func (c *Config) Set(key string, value interface{}) {
	config := (toml.Tree)(*c)
	config.Set(key, value)
	*c = (Config)(config)
}

// Save writes the config to the specified path
func (c Config) Save(path string) (int64, error) {
	config := (toml.Tree)(c)
	output, err := config.Marshal()
	if err != nil {
		return 0, fmt.Errorf("unable to convert to TOML: %v", err)
	}

	n, err := engine.Config(path).Write(output)
	return int64(n), err
}
