// Copyright (c) 2023, NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//  * Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
//  * Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  * Neither the name of NVIDIA CORPORATION nor the names of its
//    contributors may be used to endorse or promote products derived
//    from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
// OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#pragma once

#include "client_backend/client_backend.h"
#include "constants.h"
#include "data_loader.h"
#include "infer_data.h"
#include "infer_data_manager_base.h"
#include "model_parser.h"
#include "perf_utils.h"

namespace triton { namespace perfanalyzer {

/// Manages infer data to prepare an inference request and the resulting
/// inference output from triton server
class InferDataManager : public InferDataManagerBase {
 public:
  InferDataManager(
      const size_t max_threads, const int32_t batch_size,
      const std::shared_ptr<ModelParser>& parser,
      const std::shared_ptr<cb::ClientBackendFactory>& factory,
      const std::shared_ptr<DataLoader>& data_loader)
      : max_threads_(max_threads),
        InferDataManagerBase(batch_size, parser, factory, data_loader)
  {
  }

  /// Initialize this object. Must be called before any other functions
  /// \return cb::Error object indicating success or failure.
  cb::Error Init() override;

 protected:
  const size_t max_threads_{1};
  std::map<std::tuple<size_t, std::string, int, int>, cb::InferInput*> inputs_;

  cb::Error CreateAndPopulateInputs();
  cb::Error CreateAndPopulateInput(
      const size_t thread_id, const std::string& name,
      const ModelTensor& model_tensor, int stream_id, int step_id);

  cb::InferInput* GetInput(
      const size_t thread_id, const std::string& name, int stream_id,
      int step_id);

  cb::Error InitInferDataInput(
      const std::string& name, const ModelTensor& model_tensor,
      InferData& infer_data) override;

  cb::Error InitInferDataOutput(
      const std::string& name, InferData& infer_data) override;

  /// Helper function to update the inputs
  /// \param thread_id The ID of the calling thread
  /// \param stream_index The data stream to use for next data
  /// \param step_index The step index to use for next data
  /// \param infer_data The target InferData object
  /// \return cb::Error object indicating success or failure.
  cb::Error UpdateInputs(
      const size_t thread_id, const int stream_index, const int step_index,
      InferData& infer_data);

#ifndef DOCTEST_CONFIG_DISABLE
 public:
  InferDataManager() = default;
#endif
};

}}  // namespace triton::perfanalyzer
