# 分类器



本示例通过ResNet50模型说明如何使用MIGraphX C++ API进行分类模型的推理，包括如何预处理、推理并获取推理结果。



## 模型简介

本示例使用了经典的ResNet50模型，onnx文件在Resource/Models/文件夹下，模型结构可以通过netron (https://netron.app/) 查看，该模型的输入shape为[1,3,224,224] ，数据排布为NCHW。

​																														

## 预处理

在将数据输入到模型之前，需要对图像做如下预处理操作：

- 图像格式转换，BGR转换为RGB

- 调整图像大小，并在中心窗口位置裁剪出224x224大小的图像

- normalize操作，对图像减均值再除方差

- 转换数据排布为NCHW

  

本示例代码主要采用了OpenCV实现了预处理操作：

```c++
ErrorCode Classifier::Classify(const std::vector<cv::Mat> &srcImages,std::vector<std::vector<ResultOfPrediction>> &predictions)
{
	...
	
    // 预处理
    std::vector<cv::Mat> image;
    for(int i =0;i<srcImages.size();++i)
    {
        //BGR转换为RGB
        cv::Mat imgRGB;
        cv::cvtColor(srcImages[i], imgRGB, cv::COLOR_BGR2RGB);

        // 调整大小，保持长宽比
        cv::Mat shrink;
        float ratio = (float)256 / min(imgRGB.cols, imgRGB.rows);
        if(imgRGB.rows > imgRGB.cols)
        {
            cv::resize(imgRGB, shrink, cv::Size(256, int(ratio * imgRGB.rows)), 0, 0);
        }
        else
        {
            cv::resize(imgRGB, shrink, cv::Size(int(ratio * imgRGB.cols), 256), 0, 0);
        }

        // 裁剪中心窗口
        int start_x = shrink.cols/2 - 224/2;
        int start_y = shrink.rows/2 - 224/2;
        cv::Rect rect(start_x, start_y, 224, 224);
        cv::Mat images = shrink(rect);
        image.push_back(images);
    }

    // normalize并转换为NCHW
    cv::Mat inputBlob;
    Image2BlobParams image2BlobParams;
    image2BlobParams.scalefactor=cv::Scalar(1/58.395, 1/57.12, 1/57.375);
    image2BlobParams.mean=cv::Scalar(123.675, 116.28, 103.53);
    image2BlobParams.swapRB=false;
    blobFromImagesWithParams(image,inputBlob,image2BlobParams);
                    
     ...
}
```

blobFromImagesWithParams()函数支持多个输入图像，首先对输入图像各通道减对应的均值，然后乘以各通道对应的缩放系数，最后转换为NCHW，最终将转换好的数据保存到inputBlob中，然后就可以输入到模型中执行推理了。

## 量化

该示例工程提供了fp16和int8两种量化方法，可以在Resource/Configuration.xml文件中设置是否需要量化：

```c++
<opencv_storage>
	<!--分类器-->
	<Classifier>
		...
		<UseInt8>0</UseInt8>      // 设置为1时，开启INT8量化
		<UseFP16>0</UseFP16>      // 设置为1时，开启FP16量化
        ...
	</Classifier>
</opencv_storage>
```

### FP16量化

使用FP16模式只需要在编译前调用migraphx::quantize_fp16() 即可。

```c++
migraphx::quantize_fp16(net);
```

### INT8量化

使用INT8模式需要提供量化校准数据，MIGraphX采用线性量化算法，通过校准数据计算量化参数并生成量化模型。为了保证量化精度，一般使用测试集或验证集中的数据作为校准数据。

```c++
// 创建量化校准数据,建议使用测试集中的多张典型图像
cv::Mat srcImage=cv::imread("../Resource/Images/ImageNet_test.jpg",1);
std::vector<cv::Mat> srcImages;
for(int i=0;i<inputShape.lens()[0];++i)
{
	srcImages.push_back(srcImage);
}

// 数据预处理
...

// 创建量化数据
std::unordered_map<std::string, migraphx::argument> inputData;
inputData[inputName]= migraphx::argument{inputShape, (float*)inputBlob.data};
std::vector<std::unordered_map<std::string, migraphx::argument>> calibrationData = {inputData};

// INT8量化
migraphx::quantize_int8(net, gpuTarget, calibrationData);
```

## 推理

完成预处理后，就可以执行推理了：

```c++
ErrorCode Classifier::Classify(const std::vector<cv::Mat> &srcImages,std::vector<std::vector<ResultOfPrediction>> &predictions)
{
	...
	
	// 预处理
	
	// 创建输入数据
    migraphx::argument result;
    if(useoffloadcopy)
    {
        std::unordered_map<std::string, migraphx::argument> inputData;
        inputData[inputName]= migraphx::argument{inputShape, (float*)inputBlob.data};

        // 推理
        std::vector<std::string> outputNames={"resnetv24_dense0_fwd"};    // 设置返回的输出节点
        std::vector<migraphx::argument> results = net.eval(inputData,outputNames);

        result  = results[0];                                             // 获取第一个输出节点的数据
    }
    else
    {
        // 为输出节点分配device内存，用于保存输出数据
        std::unordered_map<std::string, migraphx::argument> modelData=CreateOutputData(net);

        // 将输入转换为device数据
        migraphx::argument inputData=migraphx::gpu::to_gpu(migraphx::argument{inputShape, (float*)inputBlob.data});

        // 使用device数据作为输入数据,inputData.data()返回的是device地址
        modelData[inputName]= migraphx::argument{inputShape, inputData.data()};

        // 推理
        std::vector<std::string> outputNames={"resnetv24_dense0_fwd"};    // 设置返回的输出节点
        std::vector<migraphx::argument> results = net.eval(modelData,outputNames);

        result  = migraphx::gpu::from_gpu(results[0]);                    // 将第一个输出节点的数据拷贝到host端
    }
   	
   	...
   	
}
```

- 输入数据根据是否需要数据拷贝分为两种：offload=true、offload=false。当offload为true时，执行if分支，不需要进行内存拷贝，inputData表示MIGraphX的输入数据，inputData是一个映射关系，每个输入节点名都会对应一个输入数据，如果有多个输入，则需要为每个输入节点名创建数据，inputName表示输入节点名，migraphx::argument{inputShape, (float*)inputBlob.data}表示该节点名对应的数据，这里是通过前面预处理的数据inputBlob来创建的，第一个参数表示数据的shape，第二个参数表示数据指针。当offload为false时，执行else分支，需要进行内存拷贝，为输入节点和输出节点分配device内存，用来保存输入数据和输出数据，使用device数据作为输入数据，执行推理。
- net.eval(inputData)返回模型的推理结果，由于这里只有一个输出节点，所以std::vector中只有一个数据，results[0]表示第一个输出节点，这里对应resnetv24_dense0_fwd节点，获取输出数据。

另外，如果想要指定输出节点，可以在eval()方法中通过提供outputNames参数来实现：

```c++
...
// 推理
std::vector<std::string> outputNames = {"resnetv24_dense0_fwd"};
std::vector<migraphx::argument> results = net.eval(inputData, outputNames);
...
```

- 如果没有指定outputName参数，则默认输出所有输出节点，此时输出节点的顺序与ONNX中输出节点顺序保持一致，可以通过netron查看ONNX文件的输出节点的顺序。  