# Copyright (c) 2021-2022, NVIDIA CORPORATION. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#           http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from collections import defaultdict, deque
import datetime
import errno
import os
import time

import torch
import torch.distributed as dist


class ScratchPad:
    target_n = None
    target_labels_padded = None
    target_boxes_padded = None
    target_matched_idxs = None
    gt_classes_target = None
    batch_size_vector = None


class SmoothedValue(object):
    """Track a series of values and provide access to smoothed values over a
    window or the global series average.
    """

    def __init__(self, window_size=20, fmt=None):
        if fmt is None:
            fmt = "{median:.4f} ({global_avg:.4f})"
        self.deque = deque(maxlen=window_size)
        self.total = 0.0
        self.count = 0
        self.fmt = fmt

    def update(self, value, n=1):
        self.deque.append(value)
        self.count += n
        self.total += value * n

    def synchronize_between_processes(self, group=None):
        """
        Warning: does not synchronize the deque!
        """
        if not is_dist_avail_and_initialized():
            return
        t = torch.tensor([self.count, self.total], dtype=torch.float64, device='cuda')
        dist.barrier(group=group)
        dist.all_reduce(t, group=group)
        t = t.tolist()
        self.count = int(t[0])
        self.total = t[1]

    @property
    def median(self):
        d = torch.tensor(list(self.deque))
        return d.median().item()

    @property
    def avg(self):
        d = torch.tensor(list(self.deque), dtype=torch.float32)
        return d.mean().item()

    @property
    def global_avg(self):
        return self.total / self.count

    @property
    def max(self):
        return max(self.deque)

    @property
    def value(self):
        return self.deque[-1]

    def __str__(self):
        return self.fmt.format(
            median=self.median,
            avg=self.avg,
            global_avg=self.global_avg,
            max=self.max,
            value=self.value)


def all_gather(data, group):
    """
    Run all_gather on arbitrary picklable data (not necessarily tensors)
    Args:
        data: any picklable object
    Returns:
        list[data]: list of data gathered from each rank
    """
    world_size = group.size() if group else get_world_size()
    if world_size == 1:
        return [data]
    data_list = [None] * world_size
    dist.all_gather_object(object_list=data_list, obj=data, group=group)
    return data_list


def broadcast(data, src, group):
    """
    Run broadcast on arbitrary picklable data (not necessarily tensors)
    Args:
        data: any picklable object
        src: Source rank from which to broadcast data
    Returns:
        list[data]: list of data gathered from each rank
    """
    world_size = group.size() if group else get_world_size()
    if world_size == 1:
        return data
    data_list = data if isinstance(data, list) else [data]
    dist.broadcast_object_list(object_list=data_list, src=src, group=group)
    return data_list if isinstance(data, list) else data_list[0]


def reduce_dict(input_dict, group, average=True):
    """
    Args:
        input_dict (dict): all the values will be reduced
        average (bool): whether to do average or sum
    Reduce the values in the dictionary from all processes so that all processes
    have the averaged results. Returns a dict with the same fields as
    input_dict, after reduction.
    """
    world_size = group.size() if group else get_world_size()
    if world_size < 2:
        return input_dict
    with torch.no_grad():
        names = []
        values = []
        # sort the keys so that they are consistent across processes
        for k in sorted(input_dict.keys()):
            names.append(k)
            values.append(input_dict[k])
        values = torch.stack(values, dim=0)
        dist.all_reduce(tensor=values, group=group)
        if average:
            values /= world_size
        reduced_dict = {k: v for k, v in zip(names, values)}
    return reduced_dict


class SimpleTimer(object):
    def __init__(self, prefix=""):
        self.prefix = prefix

    def __enter__(self):
        self.start = time.time()
        return self

    def __exit__(self, *args):
        self.end = time.time()
        run_time = self.end - self.start
        print(f"{self.prefix}{run_time}")


class MetricLogger(object):
    def __init__(self, delimiter="\t"):
        self.meters = defaultdict(SmoothedValue)
        self.delimiter = delimiter
        self.summary = defaultdict(lambda: None)
        self.current_iter = 0

    def update(self, **kwargs):
        for k, v in kwargs.items():
            if isinstance(v, torch.Tensor):
                v = v.item()
            assert isinstance(v, (float, int))
            self.meters[k].update(v)

    def __getattr__(self, attr):
        if attr in self.meters:
            return self.meters[attr]
        if attr in self.__dict__:
            return self.__dict__[attr]
        raise AttributeError("'{}' object has no attribute '{}'".format(
            type(self).__name__, attr))

    def __str__(self):
        loss_str = []
        for name, meter in self.meters.items():
            loss_str.append(
                "{}: {}".format(name, str(meter))
            )
        return self.delimiter.join(loss_str)

    def synchronize_between_processes(self, group=None):
        for meter in self.meters.values():
            meter.synchronize_between_processes(group=group)

    def add_meter(self, name, meter):
        self.meters[name] = meter

    def log_every(self, iterable, print_freq, header=None):
        self.current_iter = 0
        self.summary['samples'] = 0
        if not header:
            header = ''
        start_time = time.time()
        self.summary['start_time'] = start_time
        end = time.time()
        iter_time = SmoothedValue(fmt='{avg:.4f}')
        data_time = SmoothedValue(fmt='{avg:.4f}')
        space_fmt = ':' + str(len(str(len(iterable)))) + 'd'
        if torch.cuda.is_available():
            log_msg = self.delimiter.join([
                header,
                '[{0' + space_fmt + '}/{1}]',
                'eta: {eta}',
                '{meters}',
                'time: {time}',
                'data: {data}',
                'max mem: {memory:.0f}'
            ])
        else:
            log_msg = self.delimiter.join([
                header,
                '[{0' + space_fmt + '}/{1}]',
                'eta: {eta}',
                '{meters}',
                'time: {time}',
                'data: {data}'
            ])
        MB = 1024.0 * 1024.0
        for obj in iterable:
            data_time.update(time.time() - end)
            yield obj
            iter_time.update(time.time() - end)
            if self.current_iter % print_freq == 0 or self.current_iter == len(iterable) - 1:
                eta_seconds = iter_time.global_avg * (len(iterable) - self.current_iter)
                eta_string = str(datetime.timedelta(seconds=int(eta_seconds)))
                if torch.cuda.is_available():
                    print(log_msg.format(
                        self.current_iter, len(iterable), eta=eta_string,
                        meters=str(self),
                        time=str(iter_time), data=str(data_time),
                        memory=torch.cuda.max_memory_allocated() / MB))
                else:
                    print(log_msg.format(
                        self.current_iter, len(iterable), eta=eta_string,
                        meters=str(self),
                        time=str(iter_time), data=str(data_time)))
            self.current_iter += 1
            end = time.time()
            self.summary['samples'] += len(obj[0])
            self.summary['end_time'] = end

        end_time = time.time()
        total_time = end_time - start_time
        total_time_str = str(datetime.timedelta(seconds=int(total_time)))
        print('{} Total time: {} ({:.4f} s / it)'.format(
            header, total_time_str, total_time / len(iterable)))


def collate_fn(batch):
    return tuple(zip(*batch))


def warmup_lr_scheduler(optimizer, start_iter, warmup_iters, warmup_factor):

    def f(x):
        x = x + start_iter
        if x >= warmup_iters:
            return 1
        alpha = float(x) / warmup_iters
        return warmup_factor * (1 - alpha) + alpha

    return torch.optim.lr_scheduler.LambdaLR(optimizer, f)


def mkdir(path):
    try:
        os.makedirs(path)
    except OSError as e:
        if e.errno != errno.EEXIST:
            raise


def setup_for_distributed(is_master):
    """
    This function disables printing when not in master process
    """
    import builtins as __builtin__
    builtin_print = __builtin__.print

    def print(*args, **kwargs):
        force = kwargs.pop('force', False)
        if is_master or force:
            builtin_print(*args, **kwargs)

    __builtin__.print = print


def is_dist_avail_and_initialized():
    if not dist.is_available():
        return False
    if not dist.is_initialized():
        return False
    return True


def get_world_size():
    if not is_dist_avail_and_initialized():
        return 1
    return dist.get_world_size()


def get_rank():
    if not is_dist_avail_and_initialized():
        return 0
    return dist.get_rank()


def is_main_process():
    return get_rank() == 0


def save_on_master(*args, **kwargs):
    if is_main_process():
        torch.save(*args, **kwargs)


def barrier(group):
    if not is_dist_avail_and_initialized():
        return
    torch.distributed.barrier(group)


def init_distributed_mode(args):
    if args.world_size > 1 : 
        args.rank = args.local_rank
        args.world_size = args.world_size
        args.gpu = args.rank
    elif 'SLURM_PROCID' in os.environ:
        args.rank = int(os.environ['SLURM_PROCID'])
        args.gpu = args.rank % torch.cuda.device_count()
    else:
        print('Not using distributed mode')
        args.distributed = False
        args.num_train_ranks = 1
        args.num_eval_ranks = 1
        args.rank = 0
        args.ranks = 1
        args.train_ranks = [0]
        args.eval_ranks = [0]
        args.train_rank = 0
        args.eval_rank = 0
        return None, None

    args.distributed = True

    torch.cuda.set_device(args.gpu)
    args.dist_backend = 'nccl'
    print(f'| distributed init (rank {args.rank}): {args.dist_url}')
    if args.cuda_graphs or args.cuda_graphs_eval:
        os.environ["NCCL_ASYNC_ERROR_HANDLING"] = "0"
    torch.distributed.init_process_group(backend=args.dist_backend, init_method=args.dist_url,
                                         world_size=args.world_size, rank=args.rank)
    torch.distributed.barrier()

    args.ranks = list(range(args.world_size))
    if args.num_eval_ranks is None:
        args.num_train_ranks = args.world_size
        args.num_eval_ranks = args.world_size

        args.train_ranks = args.ranks
        args.eval_ranks = args.ranks
        args.train_rank = args.rank
        args.eval_rank = args.rank
    else:
        args.num_train_ranks = args.world_size - args.num_eval_ranks

        args.train_ranks = args.ranks[:args.num_train_ranks]
        args.eval_ranks = args.ranks[args.num_train_ranks:]
        args.train_rank = args.rank
        args.eval_rank = args.rank - args.num_train_ranks

    assert 1<=args.num_train_ranks<=args.world_size, "Number of training ranks must be between 1 and world size"
    assert 1<=args.num_eval_ranks<=args.world_size, "Number of validation ranks must be between 1 and world size"

    # create training and validation comm groups
    train_group = torch.distributed.new_group(ranks=args.train_ranks)
    eval_group = torch.distributed.new_group(ranks=args.eval_ranks)

    setup_for_distributed(args.train_rank==0 or args.eval_rank==0)

    # init new comms
    tmp_tensor = torch.ones([1], device='cuda')
    torch.distributed.all_reduce(tmp_tensor, group=train_group)
    torch.distributed.all_reduce(tmp_tensor, group=eval_group)

    return train_group, eval_group
