# SCCL 引导模块文档

## 概述
SCCL引导模块负责在分布式系统中跨多个节点初始化通信环境。该模块处理节点的唯一标识、通信的根节点建立以及通信结构的初始化。核心功能封装在[bootstrap.h](bootstrap.h)
头文件中，其对应的实现位于
[bootstrap.cpp](bootstrap.cpp)。辅助功能由
[bootstrap_net.cpp](bootstrap_net.cpp)
、[bootstrap_utils.cpp](bootstrap_utils.cpp)
和[physical_links.cpp](physical_links.cpp)
提供。

## 核心组件
### 1. BootstrapHandle_t 结构体
[BootstrapHandle_t](bootstrap_utils.h#26)
结构体用于存储节点的唯一识别信息，包括用于套接字通信的魔数和用于网络通信的套接字地址。

```c++
typedef struct BootstrapHandle {
    uint64_t magic;       // 随机数，用于套接字通信
    scclSocketAddress_t addr; // 地址，用于网络通信
} BootstrapHandle_t;
```

### 2. scclRankInfo_t 结构体
`scclRankInfo_t`
结构体存储每个排名的通信节点信息，包括GPU、RDMA（网络）和CPU节点。

```c++
typedef struct scclRankInfo {
    struct {
        scclSocket_t listen_sock; // 监听套接字
    } cpu;
    struct {
        int dev;
        char name[8];
        char gcn[8];
        int compCap;
        int64_t pciBusId;
        char pciPath[128];
    } gpu;
    struct {
        int count;
        char name[8];
        char pciPath[128];
        uint64_t guid;
        uint8_t ptrSupport;
        int speed;
        int port;
        float latency;
        int maxComms;
        int maxRecvs;
    } net;

    int rank;
    int localRank;
    uint64_t hostHash;
    uint64_t pidHash;
} scclRankInfo_t;
```

### 3. scclNodeInfo_t 结构体

`scclNodeInfo_t`结构体用于存储每个排名的拓扑连接信息。

```c++
typedef struct scclNodeInfo {
    scclTopoNode_t nodes[topoNodeMaxLocalNodes];
} scclNodeInfo_t;
```

### 4. BootstrapComm 结构体
`BootstrapComm`结构体存储引导通信信息，包括网络属性、排名信息及其他相关数据。
```c++
typedef struct BootstrapComm {
    void init(int rank, int nRanks, int localRank, int nLocalRanks);
    void destroy();

public:
    scclNet_t* scclNet;
    scclRankPhysSet_t* rank_phys_set;

    cpu_set_t cpuAffinity;
    int rank;
    int nRanks;
    int localRank;
    int nLocalRanks;
    int interRank;
    int nInterRanks;

    int hipDev;
    int deviceCnt;

    uint64_t magic;
} BootstrapComm_t;
```

### 5. Bootstrap 类
`Bootstrap`类封装了引导初始化过程，包括唯一ID生成、根创建和通信初始化。

```c++
class Bootstrap {
public:
    Bootstrap(const BootstrapHandle_t*, int rank, int nRanks);
    ~Bootstrap();

    scclResult_t init(BootstrapComm_t* bootstrap_comm);

    scclResult_t bootstrapAllGather(const void* src_data, void* dst_data, int data_size);

private:
    scclResult_t bootstrapRootGatherAndBroadcast(BootstrapNodeBasic_t* send_data_basic);
    scclResult_t bootstrapCommInitNodeInfo(scclNet_t* scclNet, scclRankInfo_t* rank_info);
    scclResult_t bootstrapCommAllGather(scclRankInfo_t* rank_info, scclNodeInfo_t* node_info, scclRankPhysSet_t* rank_phys_set);
    scclResult_t bootstrapNodesLink(scclNodeInfo_t* node_infos);

private:
    int rank, nRanks;
    int localRank, nLocalRanks;
    int interRank, nInterRanks;

    volatile uint32_t* abortFlag;

    const BootstrapHandle_t* root_handle;
    BootstrapNodeBasic_t* all_node_basic;

    bool socketInitDone;
    pthread_mutex_t bootstrapMutex;
    pthread_cond_t bootstrapCond;

    scclIpcSocket_t* ipcsocket;
};
```


## 功能描述

### bootstrapGetUniqueId
该函数为引导过程生成一个唯一ID，利用`BootstrapHandle_t`结构体，它初始化引导网络并为通信设置根句柄。
```c++
scclResult_t bootstrapGetUniqueId(BootstrapHandle_t* handle);
```

### bootstrapCreateRoot
负责为引导过程创建根节点。该函数设置根节点的监听套接字并启动引导通信。
```c++
scclResult_t bootstrapCreateRoot(BootstrapHandle_t* handle);
```

### Bootstrap::init
`Bootstrap`类的init方法是核心初始化函数。它设置通信环境，收集并广播排名信息，初始化节点信息，并建立节点连接。
```c++
scclResult_t Bootstrap::init(BootstrapComm_t* bootstrap_comm);
```

## 辅助模块

- [bootstrap_net.cpp](bootstrap_net.cpp)
处理与引导相关的网络操作，包括节点之间的套接字通信和数据传输。

- [bootstrap_utils.cpp](bootstrap_utils.cpp)
为引导过程提供实用函数，例如哈希、随机数据生成和PCI设备ID检索。

- [physical_links.cpp](physical_links.cpp)
定义`scclTopoNode`结构体及关联函数，用于生成和管理拓扑节点及其连接。

## 结论

SCCL引导模块为在分布式计算环境中初始化和管理通信提供了一个强大的框架。   
通过封装网络通信的复杂性、唯一ID生成和拓扑节点管理，它简化了可扩展和高效通信集合的开发。
