#ifndef XML_H_
#define XML_H_

#include <stdlib.h>

#include "base.h"

namespace sccl {
namespace hardware {
namespace topology {
namespace topo {

///////////////////////////////////////// 基础struct /////////////////////////////////////////

// A few constraints to make the implementation easy
#define MAX_STR_LEN 255
#define MAX_ATTR_COUNT 16
#define MAX_SUBS 32
#define MAX_NODES 1024

typedef enum node_type {
    NODE_TYPE_NONE   = 0,
    NODE_TYPE_OPEN   = 1,
    NODE_TYPE_CLOSE  = 2,
    NODE_TYPE_SINGLE = 3
} node_type_t;

// 定义一个结构体 scclXmlNode，用于表示XML节点
struct scclXmlNode {
    char name[MAX_STR_LEN + 1]; // 节点名称
    struct {
        char key[MAX_STR_LEN + 1];      // 属性键
        char value[MAX_STR_LEN + 1];    // 属性值
    } attrs[MAX_ATTR_COUNT + 1];        // 需要额外的一个来消耗额外参数
    int nAttrs;                         // 属性数量
    int type;                           // 节点类型
    struct scclXmlNode* parent;         // 父节点指针
    struct scclXmlNode* subs[MAX_SUBS]; // 子节点指针数组
    int nSubs;                          // 子节点数量
};

// 定义了一个结构体 scclXml，用于表示XML文档的结构
struct scclXml {
    struct scclXmlNode nodes[MAX_NODES]; // 节点数组，每个节点代表XML中的一个元素
    int maxIndex;                        // 当前XML结构中最大节点索引
};

struct kvDict {
    const char* str;
    int value;
};

typedef union {
    hipDeviceArch_t arch;
    int value;
    static_assert(sizeof(hipDeviceArch_t) == sizeof(int), "value must be the same size of hipDeviceArch_t.");
} scclHipDeviceArch_t;

///////////////////////////////////////// File functions /////////////////////////////////////////
#define SCCL_TOPO_XML_VERSION 2
#define SCCL_GRAPH_XML_VERSION 1

// 从文件中获取XML拓扑结构
scclResult_t scclTopoGetXmlFromFile(const char* xmlTopoFile, struct scclXml* xml, int warn);
// 将XML拓扑结构保存到文件中
scclResult_t scclTopoDumpXmlToFile(const char* xmlTopoFile, struct scclXml* xml);

// 从文件中获取XML图形结构
scclResult_t scclTopoGetXmlGraphFromFile(const char* xmlGraphFile, struct scclXml* xml);

/* 自动检测功能 */
// 根据总线ID填充GPU信息到XML结构中
scclResult_t scclTopoFillGpu(struct scclXml* xml, const char* busId, struct scclXmlNode** gpuNode);
// 根据PCI路径和网络名称填充网络信息到XML结构中
scclResult_t scclTopoFillNet(struct scclXml* xml, const char* pciPath, const char* netName, struct scclXmlNode** netNode);

/* 移除不需要的部分 */
// 修剪XML结构，移除不需要的部分
scclResult_t scclTopoTrimXml(struct scclXml* xml);

// 从系统路径中获取字符串值
scclResult_t scclTopoGetStrFromSys(const char* path, const char* fileName, char* strValue);

/**************/
/* XML Struct */
/* Functions  */
/**************/

// 获取XML节点的属性索引
scclResult_t xmlGetAttrIndex(struct scclXmlNode* node, const char* attrName, int* index);

// 获取XML节点的属性值，返回为字符串
scclResult_t xmlGetAttr(struct scclXmlNode* node, const char* attrName, const char** value);

// 获取XML节点的属性值，返回为字符串（与xmlGetAttr类似）
scclResult_t xmlGetAttrStr(struct scclXmlNode* node, const char* attrName, const char** value);

// 获取XML节点的属性值，返回为整数
scclResult_t xmlGetAttrInt(struct scclXmlNode* node, const char* attrName, int* value);

// 获取XML节点的属性值，返回为整数，如果属性不存在则返回默认值
scclResult_t xmlGetAttrIntDefault(struct scclXmlNode* node, const char* attrName, int* value, int defaultValue);

// 初始化XML节点的整数属性
scclResult_t xmlInitAttrInt(struct scclXmlNode* node, const char* attrName, const int value);

// 初始化XML节点的无符号64位整数属性
scclResult_t xmlInitAttrUint64(struct scclXmlNode* node, const char* attrName, const uint64_t value);

// 获取XML节点的属性值，返回为浮点数
scclResult_t xmlGetAttrFloat(struct scclXmlNode* node, const char* attrName, float* value);

// 初始化XML节点的浮点数属性
scclResult_t xmlInitAttrFloat(struct scclXmlNode* node, const char* attrName, const float value);

// 在XML中查找指定标签名的节点
scclResult_t xmlFindTag(struct scclXml* xml, const char* tagName, struct scclXmlNode** node);

// 在XML中查找指定标签名和属性值的节点
scclResult_t xmlFindTagKv(struct scclXml* xml, const char* tagName, struct scclXmlNode** node, const char* attrName, const char* attrValue);

// 设置XML节点的属性值
scclResult_t xmlSetAttr(struct scclXmlNode* node, const char* attrName, const char* value);

// 如果属性未设置，则设置XML节点的属性值
scclResult_t xmlSetAttrIfUnset(struct scclXmlNode* node, const char* attrName, const char* value);

// 设置XML节点的属性值为整数
scclResult_t xmlSetAttrInt(struct scclXmlNode* node, const char* attrName, const int value);

// 设置XML节点的属性值为浮点数
scclResult_t xmlSetAttrFloat(struct scclXmlNode* node, const char* attrName, const float value);

// 移除XML节点的属性
scclResult_t xmlUnsetAttr(struct scclXmlNode* node, const char* attrName);

// 获取XML节点的子节点
scclResult_t xmlGetSub(struct scclXmlNode* node, const char* subName, struct scclXmlNode** sub);

// 获取XML节点的子节点，子节点需匹配指定属性值
scclResult_t xmlGetSubKv(struct scclXmlNode* node, const char* subName, struct scclXmlNode** sub, const char* attrName, const char* attrValue);

// 获取XML节点的子节点，子节点需匹配指定整数属性值
scclResult_t xmlGetSubKvInt(struct scclXmlNode* node, const char* subName, struct scclXmlNode** sub, const char* attrName, const int attrValue);

// 在XML中添加新节点
scclResult_t xmlAddNode(struct scclXml* xml, struct scclXmlNode* parent, const char* subName, struct scclXmlNode** sub);

// 从XML中移除节点
scclResult_t xmlRemoveNode(struct scclXmlNode* node);

// 字符串到整数的转换字典，最后一个元素的str应为NULL
// 将字符串转换为整数
scclResult_t kvConvertToInt(const char* str, int* value, struct kvDict* dict);

// 将整数转换为字符串
scclResult_t kvConvertToStr(int value, const char** str, struct kvDict* dict);

} // namespace topo
} // namespace topology
} // namespace hardware
} // namespace sccl

#endif
