#pragma once

#include <stdint.h>
#include "base.h"

namespace sccl {
namespace hardware {
namespace net {

struct netIf {       // 网络接口结构体
    char prefix[64]; // 网络前缀
    int port;        // 端口号
};

// 解析字符串列表，将结果存储在网络接口列表中
int parseStringList(const char* string, struct netIf* ifList, int maxList);

// 根据给定的字符串和端口，匹配网络接口列表中的接口
bool matchIfList(const char* string, int port, struct netIf* ifList, int listSize, bool matchExact);

scclResult_t rocmLibraryInit(void);

////////////////////////////////// 用于定义网络设备 //////////////////////////////////
typedef struct {
    char* name;     // 主要用于日志记录。
    char* pciPath;  // PCI设备在/sys中的路径。
    uint64_t guid;  // NIC芯片的唯一标识符。对于具有多个PCI功能（物理或虚拟）的卡非常重要。
    int ptrSupport; // [SCCL_PTR_HOST|SCCL_PTR_CUDA|SCCL_PTR_DMABUF]
    int speed;      // 端口速度，单位为Mbps。
    int port;       // 端口号。
    float latency;  // 网络延迟
    int maxComms;   // 我们可以创建的最大通信数量
    int maxRecvs;   // 最大分组接收数量。
} scclNetProperties_t;

typedef struct {
    // 网络的名称（主要用于日志）
    const char* name;
    // 初始化网络。
    scclResult_t (*init)();
    // 返回适配器的数量。
    scclResult_t (*devices)(int* ndev);
    // 获取各种设备属性。
    scclResult_t (*getProperties)(int dev, scclNetProperties_t* props);
    // 创建一个接收对象并提供一个句柄以连接到它。该句柄最多可以是 SCCL_NET_HANDLE_MAXSIZE 字节，并将在排名之间交换以创建连接。
    scclResult_t (*listen)(int dev, void* handle, void** listenComm);
    // 连接到一个句柄并返回一个发送 comm 对象给该对等体。
    // 此调用不应阻塞以建立连接，而应成功返回 sendComm == NULL，并期望再次调用直到 sendComm != NULL。
    scclResult_t (*connect)(int dev, void* handle, void** sendComm);
    // 在远程对等体调用 connect 后最终确定连接建立。
    // 此调用不应阻塞以建立连接，而应成功返回 recvComm == NULL，并期望再次调用直到 recvComm != NULL。
    scclResult_t (*accept)(void* listenComm, void** recvComm);
    // 注册/注销内存。Comm 可以是 sendComm 或 recvComm。
    // 类型是 SCCL_PTR_HOST 或 SCCL_PTR_CUDA。
    scclResult_t (*regMr)(void* comm, void* data, int size, int type, void** mhandle);
    /* DMA-BUF 支持 */
    scclResult_t (*regMrDmaBuf)(void* comm, void* data, size_t size, int type, uint64_t offset, int fd, void** mhandle);
    scclResult_t (*deregMr)(void* comm, void* mhandle);
    // 异步发送到对等体。
    // 如果调用不能执行（或会阻塞），则可能返回 request == NULL
    scclResult_t (*isend)(void* sendComm, void* data, int size, int tag, void* mhandle, void** request);
    // 异步从对等体接收。 如果调用不能执行（或会阻塞），则可能返回 request == NULL
    scclResult_t (*irecv)(void* recvComm, int n, void** data, int* sizes, int* tags, void** mhandles, void** request);
    // 执行刷新/栅栏操作，以确保所有使用 SCCL_PTR_CUDA 接收到的数据对 GPU 可见
    scclResult_t (*iflush)(void* recvComm, int n, void** data, int* sizes, void** mhandles, void** request);
    // 测试请求是否完成。如果 size 不为 NULL，则返回发送/接收的字节数。
    scclResult_t (*test)(void* request, int* done, int* sizes);
    // 关闭并释放 send/recv comm 对象
    scclResult_t (*closeSend)(void* sendComm);
    scclResult_t (*closeRecv)(void* recvComm);
    scclResult_t (*closeListen)(void* listenComm);
} scclNet_t;

////////////////////////////////// 其他定义 //////////////////////////////////

typedef enum sccl_ptr {
    SCCL_PTR_HOST   = 0x1,
    SCCL_PTR_CUDA   = 0x2,
    SCCL_PTR_DMABUF = 0x4
} sccl_ptr_t;

#define SCCL_NET_HANDLE_MAXSIZE 128

} // namespace net
} // namespace hardware
} // namespace sccl
