#pragma once

#include <stdint.h>
#include "base.h"
#include "net_utils.h"
#include "device/net_ib.h"
#include "host/net_socket.h"

namespace sccl {
namespace hardware {
namespace net {

//////////////////////////////////
typedef enum net_type : uint8_t {
    NET_IB     = 0,
    NET_SOCKET = 1
} net_type_t;

//////////////////////////////////
inline scclResult_t initNetSpecial(scclNet_t* net) {
    int ndev;
    // 初始化网络，如果初始化失败则返回内部错误
    if(net->init() != scclSuccess)
        return scclInternalError;

    // 获取设备数量，如果获取失败则返回内部错误
    if(net->devices(&ndev) != scclSuccess)
        return scclInternalError;

    // 如果设备数量小于或等于0，则返回系统错误
    if(ndev <= 0)
        return scclSystemError;
    return scclSuccess;
}

/**
 * 初始化网络设备
 *
 * @param net 指向scclNet_t结构体的指针，表示要初始化的网络设备
 * @return scclResult_t 返回操作结果：
 *                      - scclSuccess: 初始化成功
 *                      - scclInternalError: 网络初始化或获取设备数量失败
 *                      - scclSystemError: 系统中无可用设备
 */
inline scclNet_t* initNet(net_type_t t) {
    scclNet_t* scclNet = NULL;

    if(t == NET_IB) {
        if(initNetSpecial(&(device::scclNetIb)) == scclSuccess) {
            scclNet = &(device::scclNetIb);
        }
    } else if(t == NET_SOCKET) {
        if(initNetSpecial(&(host::scclNetSocket)) == scclSuccess) {
            scclNet = &(host::scclNetSocket);
        }
    } else {
        WARN("Unsupported network type.");
    }

    return scclNet;
}

////////////////////////////////////
inline scclNet_t* scclNets[3] = {nullptr, &device::scclNetIb, &host::scclNetSocket};

} // namespace net
} // namespace hardware
} // namespace sccl
