#pragma once

#include <string.h>
#include "base.h"
#include "topo_utils.h"
#include "comm.h"
#include "rocm_smi_wrap.h"

namespace sccl {
namespace hardware {
namespace topology {
namespace bootstrap {

typedef net::net_socket::scclSocketAddress scclSocketAddress_t;
typedef net::net_socket::scclSocket scclSocket_t;
typedef net::scclNet_t scclNet_t;

// scclBootstrapHandle 结构体定义，用于存储引导句柄
struct scclBootstrapHandle {
    uint64_t magic;           // 魔术数，用于标识结构体的有效性
    scclSocketAddress_t addr; // 地址，用于网络通信
};

// 定义硬件拓扑类型枚举
typedef enum {
    GPU  = 0, // 图形处理单元
    PCI  = 1, // 外围组件互连
    XGMI = 2, // 非易失性存储器，NV卡中为nvlink
    CPU  = 3, // 中央处理器，实际上是NUMA域
    NIC  = 4, // 网络接口控制器
    NET  = 5  // 网络
} topoNodeType_t;

// 定义每个rank所持有的所有拓扑节点
struct topoLocalNode {
    struct {
        int dev;      // NVML设备编号
        char name[8]; // 设备名称
        char gcn[7];  // GCN架构名称
        int compCap;  // CUDA计算能力
    } gpu;            // GPU节点
    struct {
        scclSocketAddress_t socketAddr; // 网络地址
    } cpu;                              // CPU节点
    struct {
        net::scclNetProperties_t props;
        int count;
    } net; // 网络节点
    struct {
        int64_t busId; // PCI总线ID以int64_t格式表示
    } pci;             // pci节点
};

// 定义结构体 scclUniqueInfo，用于存储每个rank的通信节点的信息
struct scclUniqueInfo {
    struct topoLocalNode localNode;

    int rank;       // 当前节点的全局排名
    int nRanks;     // 总的节点数量
    int localRank;  // 当前节点在本地计算节点中的排名
    int localRanks; // 本地计算节点中的节点总数

    int deviceCnt;     // 设备数量
    int hipDev;        // CUDA 设备 ID
    uint64_t hostHash; // 主机哈希值
    uint64_t pidHash;  // 进程 ID 哈希值
};

struct scclUniqueInfoSet {
    int nUniqueInfo; // 通信节点的数量
    std::vector<struct scclUniqueInfo*> unique_info_vec;
};

// scclBootstrapComm 结构体定义，用于存储引导通信信息
struct scclBootstrapComm {
    scclNet_t* scclNet;

    struct scclUniqueInfo* unique_info; // 每个通信节点的基础信息

    cpu_set_t cpuAffinity; // CPU亲和性
    int WarpSize;

    void* bootstrap;              // 引导信息
    uint64_t magic;               // 魔术数，用于验证结构体
    volatile uint32_t* abortFlag; // 中止标志，非阻塞套接字设置

    // int splitShare;      // 是否使用共享内存进行分割
    // int* topParentRanks; // 顶级父节点的rank
    // /* 与代理相关的共享资源 */
    // struct scclProxyState* proxyState;
};

////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// 获取主机唯一标识的哈希值，该哈希值在裸机和容器实例中都是唯一的
uint64_t getHostHash(void);

// 获取当前进程的唯一哈希标识符
uint64_t getPidHash(void);

// 从/dev/urandom设备获取随机数据填充缓冲区
scclResult_t getRandomData(void* buffer, size_t bytes);

// 获取指定CUDA设备的PCI总线ID并转换为64位整数
scclResult_t getBusId(int hipDev, int64_t* busId);

// 获取当前HIP设备的计算能力版本号
int scclCudaCompCap(void);

// 打印唯一的拓扑信息
void printUniqueInfo(struct scclUniqueInfo* info);

} // namespace bootstrap
} // namespace topology
} // namespace hardware
} // namespace sccl
