#include <sys/types.h>
#include <unistd.h>

#include "ibvsymbols.h"

/* RDMA-core linking mode. Symbols are pointers to linked IB Verbs */
namespace sccl {
namespace hardware {
namespace net {
namespace net_ib {

#define ASSIGN_SYM(container, symbol, name) container->name = &symbol;

// Passthrough function for ibv_reg_mr macro in verbs.h
/**
 * 注册一个内存区域(MR)到保护域(PD)
 *
 * @param pd 保护域指针
 * @param addr 内存起始地址
 * @param length 内存区域长度
 * @param access 访问权限标志
 * @return 成功返回注册的MR指针，失败返回NULL
 */
struct ibv_mr* ibv_internal_reg_mr(struct ibv_pd* pd, void* addr, size_t length, int access) { return ibv_reg_mr(pd, addr, length, access); }

// Passthrough function for ibv_internal_query_port macro in verbs.h
/**
 * 查询IB设备的端口属性
 *
 * @param context IB设备上下文
 * @param port_num 端口号
 * @param port_attr 用于存储端口属性的结构体指针
 * @return 执行结果，0表示成功，非0表示失败
 *
 * @note 此函数是对ibv_query_port的简单封装
 */
int ibv_internal_query_port(struct ibv_context* context, uint8_t port_num, struct ibv_port_attr* port_attr) {
    return ibv_query_port(context, port_num, port_attr);
}

/**
 * @brief 构建IB Verbs符号表
 *
 * 该函数用于初始化IB Verbs库的函数指针，将内部实现的函数地址赋值给符号表结构体。
 * 每个IB Verbs API都对应一个内部实现函数，通过ASSIGN_SYM宏进行绑定。
 *
 * @param ibvSymbols 指向scclIbvSymbols结构体的指针，用于存储函数指针
 * @return scclResult_t 返回操作结果，成功返回scclSuccess
 *
1. ibv_get_device_list：获取当前系统中可用的InfiniBand设备列表。
2. ibv_free_device_list：释放由 ibv_get_device_list 函数返回的设备列表。
3. ibv_get_device_name：获取与特定InfiniBand设备相关联的设备名称。
4. ibv_open_device：为特定的InfiniBand设备创建一个上下文。
5. ibv_close_device：关闭特定的InfiniBand设备上下文。
6. ibv_get_async_event：获取InfiniBand设备上下文的下一个异步事件。
7. ibv_ack_async_event：确认（ack）由 ibv_get_async_event 返回的异步事件。
8. ibv_query_device：查询特定设备的信息。
9. ibv_query_gid：查询特定设备的全局唯一标识符（GUID）。
10. ibv_query_qp：查询特定队列对（QP）的信息。
11. ibv_alloc_pd：为特定上下文分配一个保护域（PD）。
12. ibv_dealloc_pd：释放由 ibv_alloc_pd 分配的保护域（PD）。
13. ibv_reg_mr_iova2：注册内存区域（MR），支持I/O虚拟地址。
14. ibv_reg_dmabuf_mr：注册内存区域（MR），使用DMabuf。
15. ibv_dereg_mr：注销内存区域（MR）。
16. ibv_create_cq：为特定上下文创建一个完成队列（CQ）。
17. ibv_destroy_cq：销毁特定的完成队列（CQ）。
18. ibv_create_qp：为特定上下文创建一个队列对（QP）。
19. ibv_modify_qp：修改特定的队列对（QP）。
20. ibv_destroy_qp：销毁特定的队列对（QP）。
21. ibv_fork_init：初始化用于多进程支持的结构。
22. ibv_event_type_str：返回描述特定事件类型的字符串。
 */
scclResult_t buildIbvSymbols(struct scclIbvSymbols* ibvSymbols) {
    ASSIGN_SYM(ibvSymbols, ibv_get_device_list, ibv_internal_get_device_list); //
    ASSIGN_SYM(ibvSymbols, ibv_free_device_list, ibv_internal_free_device_list);
    ASSIGN_SYM(ibvSymbols, ibv_get_device_name, ibv_internal_get_device_name);
    ASSIGN_SYM(ibvSymbols, ibv_open_device, ibv_internal_open_device);
    ASSIGN_SYM(ibvSymbols, ibv_close_device, ibv_internal_close_device);
    ASSIGN_SYM(ibvSymbols, ibv_get_async_event, ibv_internal_get_async_event);
    ASSIGN_SYM(ibvSymbols, ibv_ack_async_event, ibv_internal_ack_async_event);
    ASSIGN_SYM(ibvSymbols, ibv_query_device, ibv_internal_query_device);
    ASSIGN_SYM(ibvSymbols, ibv_query_gid, ibv_internal_query_gid);
    ASSIGN_SYM(ibvSymbols, ibv_query_qp, ibv_internal_query_qp);
    ASSIGN_SYM(ibvSymbols, ibv_alloc_pd, ibv_internal_alloc_pd);
    ASSIGN_SYM(ibvSymbols, ibv_dealloc_pd, ibv_internal_dealloc_pd);

    ASSIGN_SYM(ibvSymbols, ibv_reg_mr_iova2, ibv_internal_reg_mr_iova2);
    ASSIGN_SYM(ibvSymbols, ibv_reg_dmabuf_mr, ibv_internal_reg_dmabuf_mr);

    ASSIGN_SYM(ibvSymbols, ibv_dereg_mr, ibv_internal_dereg_mr);
    ASSIGN_SYM(ibvSymbols, ibv_create_cq, ibv_internal_create_cq);
    ASSIGN_SYM(ibvSymbols, ibv_destroy_cq, ibv_internal_destroy_cq);
    ASSIGN_SYM(ibvSymbols, ibv_create_qp, ibv_internal_create_qp);
    ASSIGN_SYM(ibvSymbols, ibv_modify_qp, ibv_internal_modify_qp);
    ASSIGN_SYM(ibvSymbols, ibv_destroy_qp, ibv_internal_destroy_qp);
    ASSIGN_SYM(ibvSymbols, ibv_fork_init, ibv_internal_fork_init);
    ASSIGN_SYM(ibvSymbols, ibv_event_type_str, ibv_internal_event_type_str);

    ibvSymbols->ibv_internal_reg_mr     = &ibv_internal_reg_mr;
    ibvSymbols->ibv_internal_query_port = &ibv_internal_query_port;

    return scclSuccess;
}

} // namespace net_ib
} // namespace net
} // namespace hardware
} // namespace sccl
