#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <vector> // 引入vector库
#include <thread> // 为了使用 std::this_thread::sleep_for
#include "mpi.h"
#include "net.h"
#include "ipc_socket.h"
#include "thread_pool.h"

/*
通过Unix域套接字发送和接收文件描述符的接口通常在需要在不同进程之间共享文件或网络连接时使用。
这种情况常见于需要高效通信和资源共享的高性能计算（HPC）环境、服务器进程、或者需要在进程间传递复杂数据结构的应用中。
例如，一个进程可能打开了一个到远程服务器的网络连接，然后需要将这个连接传递给另一个进程来处理后续的通信。

使用场景

- 高性能计算（HPC）和并行计算：在使用
MPI（消息传递接口）进行并行计算时，不同的进程可能需要访问相同的文件或网络资源。通过发送文件描述符，可以避免多个进程重复打开相同的文件或建立相同的网络连接，从而节省资源和时间。

-
服务器架构：在一个主进程接受客户端连接，然后将这些连接传递给工作进程处理的服务器架构中，文件描述符传递是一个有效的策略。这允许主进程继续接受新的连接，而不被单个连接的处理所阻塞。

- 插件或模块化系统：在一个进程中加载的插件可能需要访问主进程已经打开的文件或网络连接。通过文件描述符传递，插件可以直接使用这些资源，而无需重新打开或建立连接。
*/

using namespace sccl;

typedef class sccl::hardware::net::ipc_socket::scclIpcSocket scclIpcSocket_t;

template <typename T>
void send_fd(T* ipcsocket, int fd, int dst_rank) {
    if(ipcsocket->scclIpcSocketSendFd(fd, dst_rank) != scclSuccess) {
        perror("Failed to send file descriptor");
        close(fd);
        MPI_Abort(MPI_COMM_WORLD, 1);
    }
}

int main(int argc, char* argv[]) {
    MPI_Init(&argc, &argv);
    int rank, size;
    MPI_Comm_rank(MPI_COMM_WORLD, &rank);
    MPI_Comm_size(MPI_COMM_WORLD, &size);

    int dst_hash = 12345;
    scclIpcSocket_t ipcsocket(rank, size, dst_hash);

    if(rank == 0) {
        // 进程 0: 打开文件并发送文件描述符给所有其他进程
        int fd = open("testfile.txt", O_RDONLY);
        if(fd < 0) {
            perror("Failed to open file");
            MPI_Abort(MPI_COMM_WORLD, 1);
        }

        auto pthpool = ThreadPool(8);
        for(int i = 1; i < size; ++i) {
            auto task = std::bind(send_fd<scclIpcSocket_t>, &ipcsocket, fd, i);
            pthpool.enqueue(task);
        }

        std::this_thread::sleep_for(std::chrono::seconds(5));
        close(fd);
    } else {
        // 其他进程: 接收文件描述符并读取文件内容
        int fd;
        if(ipcsocket.scclIpcSocketRecvFd(&fd) < 0) {
            perror("Failed to receive file descriptor");
            MPI_Abort(MPI_COMM_WORLD, 1);
        }
        char buffer[256];
        ssize_t n = read(fd, buffer, sizeof(buffer) - 1);
        if(n > 0) {
            buffer[n] = '\0';
            printf("Process %d received: %s\n", rank, buffer);
        }
        close(fd);
    }

    std::this_thread::sleep_for(std::chrono::seconds(10));
    MPI_Finalize();
    return 0;
}
/*
单机执行
SCCL_DEBUG_LEVEL=ABORT SCCL_DEBUG_SUBSYS=BOOTSTRAP mpirun --allow-run-as-root -np 4 2_socket_mpi_fd_pthpool
*/
