#pragma once

#include <iostream>
#include <unordered_map>
#include <unordered_set>
#include <vector>
#include <string>
#include <array>
#include <fstream>
#include <sstream>
#include <unordered_map>
#include <vector>
#include <filesystem> // 需要C++17支持

#include "container.h"
#include "bootstrap_utils.h"

namespace sccl {
namespace hardware {
namespace topology {
namespace bootstrap {

constexpr size_t topoNodeMaxLocalNodes = 128; // 每个节点最多的node数量
constexpr size_t topoNodeMaxNeighbors  = 16;  // 每个node最多neighbor数量

namespace physical_links {

/*
用于生成硬件之间的图点连接关系

依赖于读取/sys/devices 文件夹下的文件来获取信息

例如： /sys/devices/pci0000:98/0000:98:00.0 文件夹下包含各种信息
*/
typedef struct scclTopoNode {
    // 构造函数声明
    scclTopoNode();
    scclTopoNode(const char* pciPath, int interRank, int terminalType, int hipDev = -1, int numaId = -1);
    scclTopoNode(int numaId, int interRank);

    // 添加邻近图点
    void addNeighbor(uint64_t neighborId);

    // 设置busId的字符串
    void setBusIdStr(const char* busIdStr);

    // 合并两个具有相同id、type和busIdStr的节点
    scclResult_t combineNode(struct scclTopoNode* other_node);

    // 清空节点
    void clearNode();

    // 打印图点信息
    void printNodeInfo(const char* prefix, bool printNeighbor = false);

public:
    uint64_t id;   // 图点id标志
    int type;      // 图点类型，对应 nodeType_t
    int64_t busId; // 总线ID字符串 "0000:00:00.0"，例如 "0000:98:03.7"，通过int64ToBusId和busIdToInt64变化
    int speed;     // 速度
    int width;     // 带宽

    std::array<uint64_t, topoNodeMaxNeighbors> neighbors; // 邻居图点
    size_t neighborCount;                                 // 邻居图点的数量
} scclTopoNode_t;

// 根据给定的PCI路径生成拓扑节点
scclResult_t generate_topo_nodes(const char* pciPath, int interRank, int hipDev, ByteSpanVector<scclTopoNode_t>& nodes_span);

// 根据numaId获取pci路径
std::string generate_topo_node_numa_info(int numaId);

// 输出id分解后的所有数据
void getIdComponents(
    uint64_t idToDecompose, int* interRank = nullptr, int* deviceValue = nullptr, int* terminalType = nullptr, int* hipDev = nullptr, int* numaId = nullptr);

// 获取GPU和NIC的dev对应路径
char* getGpuPciPath(int hipDev);
char* getNetPciPath(scclNet_t* scclNet, int hipDev);

// 打印拓扑节点的信息
void printTopoNode(ByteSpanArray<scclTopoNode_t>& nodes, int nodeIndex, const char* prefix);

} // namespace physical_links
} // namespace bootstrap
} // namespace topology
} // namespace hardware
} // namespace sccl
