#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#define SOCKET_PATH "/tmp/unix_domain_socket"

void err_quit(const char* msg) {
    perror(msg);
    exit(1);
}

/**
 * @brief Unix域套接字服务器示例
 *
 * 该程序实现了一个简单的Unix域数据报套接字服务器，用于接收消息和文件描述符。
 * 主要功能包括：
 * 1. 创建并绑定Unix域套接字
 * 2. 循环接收客户端消息和文件描述符
 * 3. 处理接收到的文件描述符(关闭)
 * 4. 清理套接字资源
 *
 * @note 当前实现中发送响应部分被注释掉了
 * @warning 程序使用goto语句进行资源清理，需注意流程控制
 */
int main() {
    int sockfd;
    struct sockaddr_un servaddr, cliaddr;
    struct iovec iov;
    struct msghdr msg;
    char buf[1024];
    char ctrl_buf[CMSG_SPACE(sizeof(int))];
    struct cmsghdr* cmsg;
    int received_fd;

    // 创建Unix域数据报套接字
    if((sockfd = socket(AF_UNIX, SOCK_DGRAM, 0)) < 0)
        err_quit("socket creation failed");

    // 清空服务器地址结构
    memset(&servaddr, 0, sizeof(servaddr));
    unlink(SOCKET_PATH);
    servaddr.sun_family = AF_UNIX;
    strncpy(servaddr.sun_path, SOCKET_PATH, sizeof(servaddr.sun_path) - 1);

    // 绑定套接字到地址
    if(bind(sockfd, (struct sockaddr*)&servaddr, sizeof(servaddr)) < 0)
        err_quit("bind failed");

    printf("Server is waiting for connections... sockfd=%d\n", sockfd);

    while(1) {
        // 清空缓冲区和消息头
        memset(buf, 0, sizeof(buf));
        memset(&msg, 0, sizeof(msg));

        iov.iov_base       = buf;
        iov.iov_len        = sizeof(buf);
        msg.msg_name       = &cliaddr;
        msg.msg_namelen    = sizeof(cliaddr);
        msg.msg_iov        = &iov;
        msg.msg_iovlen     = 1;
        msg.msg_control    = ctrl_buf;
        msg.msg_controllen = sizeof(ctrl_buf);

        // 接收消息
        if(recvmsg(sockfd, &msg, 0) < 0)
            err_quit("recvmsg failed");

        printf("Received message: %s\n", buf);

        // 处理辅助数据
        for(cmsg = CMSG_FIRSTHDR(&msg); cmsg != NULL; cmsg = CMSG_NXTHDR(&msg, cmsg)) {
            if(cmsg->cmsg_level == SOL_SOCKET && cmsg->cmsg_type == SCM_RIGHTS) {
                memmove(&received_fd, CMSG_DATA(cmsg), sizeof(received_fd));
                printf("Received file descriptor: %d\n", received_fd);
                close(received_fd); // 关闭接收到的文件描述符
            }
            goto final;
        }

        /*
        printf("111\n");
        // 发送响应
        strcpy(buf, "Message received");
        printf("222\n");
        if(sendmsg(sockfd, &msg, 0) < 0) {
            printf("333\n");
            err_quit("sendmsg failed");
        }
        printf("444\n");
        */
    }

final:
    close(sockfd);
    unlink(SOCKET_PATH);
    return 0;
}