package docker

import (
	"context"
	"strings"
	"testing"
	"time"

	"github.com/moby/moby/client"
)

func TestRegexp(t *testing.T) {
	testData := []string{
		`13:pids:/system.slice/docker-ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389.scope
12:freezer:/system.slice/docker-ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389.scope
11:rdma:/system.slice/docker-ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389.scope
10:net_cls,net_prio:/system.slice/docker-ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389.scope
9:cpuset:/system.slice/docker-ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389.scope
8:hugetlb:/system.slice/docker-ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389.scope
7:memory:/system.slice/docker-ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389.scope
6:perf_event:/system.slice/docker-ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389.scope
5:devices:/system.slice/docker-ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389.scope
4:ioasids:/
3:blkio:/system.slice/docker-ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389.scope
2:cpu,cpuacct:/system.slice/docker-ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389.scope
1:name=systemd:/system.slice/docker-ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389.scope`,
		"0::/system.slice/docker-e6369ea11c46057bcd05cb15be33014e0220e0319bb0ca15a71b295f33025798.scope",
		`12:memory:/docker/eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b
11:devices:/docker/eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b
10:hugetlb:/docker/eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b
9:net_cls,net_prio:/docker/eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b
8:perf_event:/docker/eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b
7:cpuset:/docker/eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b
6:rdma:/docker/eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b
5:pids:/docker/eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b
4:freezer:/docker/eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b
3:blkio:/docker/eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b
2:cpu,cpuacct:/docker/eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b
1:name=systemd:/docker/eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b`,
	}

	testResult := []string{
		"ce3d84d6b51029f2eb89a6ab15ccdd9756add41032945a75b58b974aeb365389",
		"e6369ea11c46057bcd05cb15be33014e0220e0319bb0ca15a71b295f33025798",
		"eb30ba674bdffab5c7165ec4c3d69cd9ebb0805670be91b4d080eaf818192d7b",
	}
	for i, s := range testData {
		lines := strings.Split(s, "\n")
		var target string
		if len(lines) > 1 {
			for _, line := range lines {
				if strings.Contains(line, "pids") {
					target = line
					break
				}
			}
		} else {
			target = lines[0]
		}
		if ReDocker.MatchString(target) {
			fields := ReDocker.FindStringSubmatch(target)
			t.Logf("fields: %v", fields)
			if fields[1] != testResult[i] {
				t.Errorf("not match: input str %s, target str: %s", target, testResult[i])
			}
		}
	}
}

func TestDocker(t *testing.T) {
	v, err := GetDockerAPIVersion()
	if err != nil {
		t.Error(err)
	}
	cli, err := client.NewClientWithOpts(client.FromEnv, client.WithVersion(v))
	if err != nil {
		t.Error(err)
	}
	defer func() {
		_ = cli.Close()
	}()
	listOpt := client.ContainerListOptions{All: true}
	//filter := make(client.Filters)
	//filter.Add()
	//listOpt.Filters = filter
	c, err := cli.ContainerList(context.Background(), listOpt)
	if err != nil {
		t.Error(err)
	}
	for _, container := range c {
		t.Logf("%+v\n", container)
	}

}

func TestGetProcessIdInDocker(t *testing.T) {
	now := time.Now()
	err := ContainerInfo.Update()
	if err != nil {
		t.Error(err)
	}
	pids, err := ContainerInfo.GetProcessIdInDocker(false)
	d := time.Since(now)
	if err != nil {
		t.Error(err)
	}
	t.Logf("%d ms", d.Milliseconds())
	for k, v := range pids {
		t.Logf("======> container %s has %d processes", k, len(v))
		for _, pidInfo := range v {
			t.Logf("  %+v", pidInfo)
		}
	}
}

func TestGetDockerAPIVersion(t *testing.T) {
	v, err := GetDockerAPIVersion()
	if err != nil {
		t.Error(err)
	}
	t.Log(v)
}

func TestTop(t *testing.T) {
	cli, err := GetDockerClient()
	if err != nil {
		t.Error(err)
	}
	cs, err := cli.ContainerList(context.Background(), client.ContainerListOptions{})
	if err != nil {
		cli.Close()
		t.Error(err)
	}

	for _, v := range cs {
		top, err := cli.ContainerTop(context.Background(), v.ID, nil)
		if err != nil {
			continue
		}
		t.Logf("%v", top)
	}
	cli.Close()
}

func TestTimeout(t *testing.T) {
	cli, err := GetDockerClient()
	if err != nil {
		t.Error(err)
	}
	ctx, fun := context.WithTimeout(context.Background(), time.Nanosecond*100)
	cs, err := cli.ContainerList(ctx, client.ContainerListOptions{})
	if err != nil {
		fun()
		cli.Close()
		t.Error(err)
	}
	fun()

	for _, v := range cs {
		top, err := cli.ContainerTop(context.Background(), v.ID, nil)
		if err != nil {
			continue
		}
		t.Logf("%v", top)
	}
	cli.Close()
}
