package main

import (
	"fmt"
	"get-container/cmd/opsflow/backend"
	"get-container/cmd/opsflow/web"
	"get-container/utils"
	"log"

	"github.com/spf13/pflag"
)

var (
	flagPort   = pflag.Int16P("port", "p", 10880, "listen port for service")
	flagServer = pflag.BoolP("server", "s", false, "run as server mode")
	flagCmd    = pflag.StringP("cmd", "c", "all", "command to execute, sys/dcu/login/all")
	flagHelp   = pflag.BoolP("help", "h", false, "show help message")
)

func main() {
	pflag.Parse()
	if *flagHelp {
		fmt.Println(`this is opsflow command line tool.
Usage:
	opsflow [options]

Options:`)
		pflag.PrintDefaults()
		return
	}

	backend.Init()
	defer backend.Shutdown()
	if *flagServer {
		log.Println("start opsflow server mode")
		err := web.WebServer(fmt.Sprintf(":%d", *flagPort))
		if err != nil {
			log.Fatalf("failed to start web server: %v", err)
		}
	}
	switch *flagCmd {
	case "sys":
		PrintSysLoad()
	case "dcu":
		PrintDCUInfo()
	case "login":
		PrintLoginInfo()
	case "all":
		PrintSysLoad()
		PrintDCUInfo()
		PrintLoginInfo()
	default:
		log.Fatalf("unknown command: %s", *flagCmd)
	}
}

func PrintSysLoad() {
	load, err := backend.GetSysLoad()
	if err != nil {
		log.Fatalf("failed to get sys load: %v", err)
	}
	fmt.Println("============== sysload =================")
	fmt.Printf("CPU Usage: %.2f%%\n", load.CPUPercent)
	fmt.Printf("Load Average (1m, 5m, 15m): %.2f, %.2f, %.2f\n", load.LoadAverage1, load.LoadAverage5, load.LoadAverage15)
	memSize := utils.MemorySize{Unit: utils.Byte, Num: load.MemTotal}
	fmt.Printf("Total Memory: %s\n", memSize.HumanReadStr(1))
	fmt.Printf("Memory Usage: %.2f%%\n", load.MemUsagePercent)
	swapSize := utils.MemorySize{Unit: utils.Byte, Num: load.SwapTotal}
	fmt.Printf("Total Swap: %s\n", swapSize.HumanReadStr(1))
	fmt.Printf("Swap Usage: %.2f%%\n", load.SwapUsagePercent)
	fmt.Println("")
}

func PrintLoginInfo() {
	logins, err := backend.GetOnlineUser()
	if err != nil {
		log.Fatalf("failed to get login info: %v", err)
	}
	fmt.Println("============== login info =================")
	for _, login := range logins {
		fmt.Printf("User: %s, Terminal: %s, Login From: %s, Login Time: %s\n", login.Name, login.TTY, login.LoginFrom, login.LoginTime.Format("2006-01-02 15:04:05"))
	}
	fmt.Println("")
}

func PrintDCUInfo() {
	dcus, err := backend.GetDCULoad()
	if err != nil {
		log.Fatalf("failed to get dcu info: %v", err)
	}
	fmt.Println("============== dcu info =================")
	for _, dcu := range dcus {
		memTotal := utils.MemorySize{Unit: utils.Byte, Num: dcu.MemTotal}
		memUsed := utils.MemorySize{Unit: utils.Byte, Num: dcu.MemUsed}
		fmt.Printf("DCU index: %d \n Fan speed: %s\n Temperature: %.2f%%\n Power Capture: %.2fw\n Power Capture: %.2fw\n VRAM total: %s\n VRAM used: %s\n DCUUtils: %.2f%%\n",dcu.Index, dcu.Fan, dcu.Temp,dcu.PwrCap,dcu.PwrAvg, memTotal.HumanReadStr(1), memUsed.HumanReadStr(1), dcu.DCUUTil)
	}
	fmt.Println("")
}