package tui

import (
	"fmt"
	"strings"
	"time"

	tea "github.com/charmbracelet/bubbletea"
	"github.com/charmbracelet/lipgloss"
)

type ModelHeader struct {
	t             time.Time
	DCUTopVersion string
	SMIVersion    string
	DriverVersion string
}

func NewModelHeader() *ModelHeader {
	return &ModelHeader{}
}

func (mh *ModelHeader) Init() tea.Cmd {
	mh.t = time.Now()
	return nil
}

func (mh *ModelHeader) Update(inputMsg tea.Msg) (tea.Model, tea.Cmd) {
	switch msg := inputMsg.(type) {
	case *ModelMsg:
		mh.t = msg.t
		mh.DCUTopVersion = msg.MyVersion
		mh.DriverVersion = msg.Version.DriverVersion
		mh.SMIVersion = msg.Version.SMIVersion
		return mh, nil
	}
	return mh, nil
}

const (
	Title = `
├───────────────────────────────┬──────────────────────┬──────────────────────┤
│ DCU  Name   Performance Level │ Bus-Id        DisP.A │ MIG M.           ECC │
│ Fan  Temp       Pwr:Usage/Cap │         Memory-Usage │ DCU-Util   PowerMode │
`
)

var (
	KeyStyle          = lipgloss.NewStyle().Foreground(lipgloss.Color("#842cffff")).Italic(true)
	KeyH, KeyQ        = KeyStyle.SetString("h").String(), KeyStyle.SetString("q").String()
	Space             = strings.Repeat(" ", 28)
	HeaderBorderStyle = lipgloss.NewStyle().Border(myBorder, true, true, false, true)
)

func (mh *ModelHeader) View() string {
	header := fmt.Sprintf("%s%s(Press %s for help or %s for quit)\n", mh.t.Format("2006-01-02 15:04:05"), Space, KeyH, KeyQ)
	style := lipgloss.NewStyle().Padding(0, 1)
	hyv := style.Width(18).Render(fmt.Sprintf("hytop: %s", mh.DCUTopVersion))
	drv := style.Width(35).Render(fmt.Sprintf("Driver Version: %s", mh.DriverVersion))
	smiv := style.Width(24).Render(fmt.Sprintf("SMI Version: %s", mh.SMIVersion))
	return header + HeaderBorderStyle.Render(hyv+drv+smiv) + Title
}
