package tui

import (
	"get-container/gpu"
	"time"

	tea "github.com/charmbracelet/bubbletea"
	"github.com/charmbracelet/lipgloss"
)

const (
	DCUTopVersion = "1.0.0"
)

// ModelMsg 模型信息，在父组件和各个子组件间共享信息
type ModelMsg struct {
	t         time.Time          // 当前时间
	index     uint64             // update次数
	Version   *gpu.HYVersionInfo // gpu版本相关信息
	MyVersion string
	// DCUInfo    []gpu.DCUInfo      // DCU全量信息
	// DCUPidInfo []gpu.DCUPidInfo   // 使用dcu的进程信息
}

type TickMsg time.Time

// ModelMain tui主模型类
type ModelMain struct {
	Header *ModelHeader
	// GPUInfo     *ModelDCUInfo
	// SysLoad     *ModelSysLoad
	// ProcessInfo *ModelProcess
	index    uint64    // 记录update次数的值
	modelMsg *ModelMsg // 记录模型信息
}

func NewModelMain() ModelMain {
	result := ModelMain{}
	result.Header = &ModelHeader{}
	return result
}

func tickCmd() tea.Cmd {
	return tea.Every(time.Second, func(t time.Time) tea.Msg {
		return TickMsg(t)
	})
}

func sendMsgCmd(modelMsg *ModelMsg) tea.Cmd {
	return func() tea.Msg {
		return *modelMsg
	}
}

// Init 初始化信息
func (m *ModelMain) Init() tea.Cmd {
	modelMsg := &ModelMsg{}
	if err := initModelInfo(modelMsg); err != nil {
		return tea.Quit
	}
	cmds := make([]tea.Cmd, 0)
	if c := m.Header.Init(); c != nil {
		cmds = append(cmds, c)
	}
	m.modelMsg = modelMsg
	// 将调用初始化方法
	cmds = append(cmds, tickCmd(), sendMsgCmd(modelMsg))
	return tea.Batch(cmds...)
}

func (m *ModelMain) Update(inputMsg tea.Msg) (tea.Model, tea.Cmd) {
	switch msg := inputMsg.(type) {
	case tea.KeyMsg: // 键盘事件
		switch msg.String() {
		case "ctrl+c", "q":
			return m, tea.Quit
		}
	case TickMsg: // 定时事件
		m.index++
		updateModelInfo(m.modelMsg, m.index, time.Time(msg))
		cmds := make([]tea.Cmd, 0)
		header, cmdHeader := m.Header.Update(m.modelMsg)
		m.Header = header.(*ModelHeader)
		cmds = append(cmds, cmdHeader, tickCmd())
		return m, tea.Batch(cmds...)
	}
	return m, nil
}

func (m *ModelMain) View() string {
	return m.Header.View()
}

// type ModelDCUInfo struct{}
// type ModelSysLoad struct{}
// type ModelProcess struct{}

var myBorder = lipgloss.Border{
	Top:          "═",
	TopLeft:      "╒",
	TopRight:     "╕",
	Bottom:       "═",
	BottomLeft:   "╘",
	BottomRight:  "╛",
	Left:         "│",
	Right:        "│",
	MiddleLeft:   "├",
	MiddleRight:  "┤",
	Middle:       "┼",
	MiddleTop:    "┬",
	MiddleBottom: "┴",
}

func initModelInfo(model *ModelMsg) error {
	model.MyVersion = DCUTopVersion
	model.index = 0
	model.t = time.Now()
	ver, err := gpu.GetHYVersionInfo()
	if err != nil {
		return err
	}
	model.Version = ver
	return nil
}

// updateModelInfo 更新模型信息
func updateModelInfo(modelMsg *ModelMsg, index uint64, t time.Time) {
	modelMsg.index = index
	modelMsg.t = t
}
