package utils

import (
	"fmt"
	"strings"
	"sync/atomic"
	"testing"
	"time"

	"github.com/emirpasic/gods/v2/maps/treemap"
)

func TestRegexp(t *testing.T) {
	t.Log(ReUnit.MatchString("b"))
	t.Log(ReUnit.MatchString("kb"))
	t.Log(ReUnit.MatchString("kib"))

	t.Logf("%+v, %d", ReUnit.FindStringSubmatch("b"), len(ReUnit.FindStringSubmatch("b")))
	t.Logf("%+v, %d", ReUnit.FindStringSubmatch("kb"), len(ReUnit.FindStringSubmatch("kb")))
	t.Logf("%+v, %d", ReUnit.FindStringSubmatch("kib"), len(ReUnit.FindStringSubmatch("kib")))

	if ReStorageSize.MatchString("123MiB") {
		t.Logf("%+v", ReStorageSize.FindStringSubmatch("123MiB"))
	} else {
		t.Errorf("Error match 123MiB")
	}

	if ReStorageSize.MatchString("123MB") {
		t.Logf("%+v", ReStorageSize.FindStringSubmatch("123MB"))
	} else {
		t.Errorf("Error match 123MB")
	}
}

func TestParseUnit(t *testing.T) {
	testData := []string{"MiB", "MB", "B", "b"}
	result := []StorageCapacityUnit{MiB, MB, Byte, Byte}
	for index, unit := range testData {
		u, err := ParseUnit(unit)
		if err != nil {
			t.Errorf("Error match %d: %s", index, err)
		}
		if u != result[index] {
			t.Errorf("Error match %d: expected %d, got %d", index, u, result[index])
		}
	}
}

func TestParseMemorySize(t *testing.T) {
	testData := []string{"1MiB", "2MB", "3B", "4b", "5 PiB"}
	result := []MemorySize{{Num: 1, Unit: MiB}, {Num: 2, Unit: MB}, {Num: 3, Unit: Byte}, {Num: 4, Unit: Byte}, {Num: 5, Unit: PiB}}
	for index, unit := range testData {
		u, err := ParseMemorySize(unit)
		if err != nil {
			t.Errorf("Error match %d: %s", index, err)
		}
		if u.Num != result[index].Num || u.Unit != result[index].Unit {
			t.Errorf("Error match %d: expected %d, got %d", index, u, result[index])
		}
	}
}

func TestHumanReadStr(t *testing.T) {
	ms := MemorySize{Num: 0, Unit: Byte}
	t.Log(ms.HumanReadStr(2))
	ms.Num = 1025
	ms.Unit = PiB
	t.Log(ms.HumanReadStr(1))
}

func TestPCIBusId(t *testing.T) {
	t.Log(PCIBus(0x1, 1))
	t.Log(PCIBus(0x12, 1))
	t.Log(PCIBus(0x123, 1))
	t.Log(PCIBus(0x1234, 1))
	t.Log(PCIBus(0x12345, 1))
	t.Log(PCIBus(0x123456, 1))
	t.Log(PCIBus(0x1234567, 1))
	t.Log(PCIBus(0x12345678, 1))
	t.Log(PCIBus(0x123456789, 1))
	i := atomic.Int32{}
	t.Log(i.Load())
}

func TestTimesort(t *testing.T) {
	fmt.Println("--- 打印所有 256 个 8 点盲文（Unicode 字符）---")
	fmt.Println("  (如果您的字体支持，将显示从 U+2800 到 U+28FF 的符号)")
	fmt.Println(strings.Repeat("-", 60))

	// 8点盲文的 Unicode 范围是 U+2800 到 U+28FF
	const startCodePoint = 0x2800
	const endCodePoint = 0x28FF
	const charsPerRow = 16 // 每行打印 16 个字符，共 16 行

	// 循环遍历并打印所有 256 个字符
	for i := startCodePoint; i <= endCodePoint; i++ {
		// 打印 Unicode 符号本身
		fmt.Printf("%c", rune(i))

		// 每 16 个字符换一行
		if (i-startCodePoint+1)%charsPerRow == 0 {
			fmt.Print("\n")
		} else {
			fmt.Print("|") // 字符之间用空格分隔
		}
	}

	fmt.Println(strings.Repeat("-", 60))
	fmt.Println("所有 256 个组合打印完成 (从 U+2800 到 U+28FF)。")
}

func TestChartype(t *testing.T) {
	r := GetCharType(LeftFullMW)
	t.Log(r)
	t.Log(CharTypeOr(GetCharType(' '), GetCharType(' ')))
	t.Log(GetCharType('⢠'))
	s := treemap.NewWith[time.Time, int](func(x, y time.Time) int {
		if x.Before(y) {
			return -1
		}
		if x.After(y) {
			return 1
		}
		return 0
	})

	s.Put(time.Now(), 3)
	s.Put(time.Now().Add(time.Second), 2)
	s.Put(time.Now().Add(-2*time.Second), 9)
	i := s.Iterator()
	for {
		if !i.Next() {
			break
		}
		t.Logf("%v: %d", i.Key(), i.Value())
	}
}
