package tui

import (
	"fmt"
	"strings"
	"time"

	tea "github.com/charmbracelet/bubbletea"
	"github.com/charmbracelet/lipgloss"
)

type ModelHeader struct {
	DCUTopVersion string
	SMIVersion    string
	DriverVersion string
	darkMode      bool
	strBuilder    strings.Builder
	style         lipgloss.Style
	static        string
	updateStatic  bool
}

func NewModelHeader() *ModelHeader {
	return &ModelHeader{
		darkMode:     false,
		style:        lipgloss.NewStyle().Padding(0, 1),
		updateStatic: true,
	}
}

func (mh *ModelHeader) Init() tea.Cmd {
	return nil
}

func (mh *ModelHeader) Update(inputMsg tea.Msg) (tea.Model, tea.Cmd) {
	switch msg := inputMsg.(type) {
	case *ModelMsg:
		if mh.DCUTopVersion != msg.MyVersion || mh.DriverVersion != msg.Version.DriverVersion || mh.SMIVersion != msg.Version.SMIVersion {
			mh.DCUTopVersion = msg.MyVersion
			mh.DriverVersion = msg.Version.DriverVersion
			mh.SMIVersion = msg.Version.SMIVersion
			hyv := FormatStr(fmt.Sprintf(" hytop: %s", mh.DCUTopVersion), 18, lipgloss.Left)
			drv := FormatStr(fmt.Sprintf("Driver Version: %s", mh.DriverVersion), 35, lipgloss.Left)
			smiv := FormatStr(fmt.Sprintf("SMI Version: %s", mh.SMIVersion), 24, lipgloss.Left)
			mh.static = HeaderBorderStyle.Render(hyv+drv+smiv) + Title
			mh.updateStatic = true
		} else {
			mh.updateStatic = false
		}
		return mh, nil
	case *ActionMsg:
		if msg.Action != nil && msg.VM == VMMain {
			mh.darkMode = true
		} else {
			mh.darkMode = false
		}
		return mh, nil
	}
	return mh, nil
}

const (
	Title = `
├───────────────────────────────┬──────────────────────┬──────────────────────┤
│ DCU  Name   Performance Level │ Bus-Id        DisP.A │ MIG M.           ECC │
│ Fan  Temp       Pwr:Usage/Cap │         Memory-Usage │ DCU-Util   PowerMode │`
)

var (
	KeyStyle          = lipgloss.NewStyle().Foreground(lipgloss.Color("#842cffff")).Italic(true)
	KeyLowLeightStyle = lipgloss.NewStyle().Foreground(lipgloss.Color("#6a32b992")).Italic(true)
	KeyH, KeyQ        = KeyStyle.SetString("h").String(), KeyStyle.SetString("q").String()
	KeyHL, KeyQL      = KeyLowLeightStyle.SetString("h").String(), KeyLowLeightStyle.SetString("q").String()
	Space             = strings.Repeat(" ", 28)
	HeaderBorderStyle = lipgloss.NewStyle().Border(myBorder, true, true, false, true)
)

func (mh *ModelHeader) View() string {
	mh.strBuilder.Reset()
	if mh.darkMode {
		mh.strBuilder.WriteString(time.Now().Format("2006-01-02 15:04:05"))
		mh.strBuilder.WriteString(Space)
		mh.strBuilder.WriteString("(Press ")
		header := LowLeightStyle.Render(mh.strBuilder.String())
		mh.strBuilder.Reset()
		mh.strBuilder.WriteString(header)
		mh.strBuilder.WriteString(KeyHL)
		mh.strBuilder.WriteString(LowLeightStyle.Render(" for help or "))
		mh.strBuilder.WriteString(KeyQL)
		mh.strBuilder.WriteString(LowLeightStyle.Render(" for quit)"))
		mh.strBuilder.WriteByte('\n')
		header = mh.strBuilder.String()
		return header + LowLeightStyle.SetString(mh.static).String() + "\n"
	} else {
		mh.strBuilder.WriteString(fmt.Sprintf("%s%s(Press %s for help or %s for quit)\n", time.Now().Format("2006-01-02 15:04:05"), Space, KeyH, KeyQ))
		mh.strBuilder.WriteString(mh.static)
		mh.strBuilder.WriteByte('\n')
		return mh.strBuilder.String()
	}
}
