/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2015-2018 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "profiling.H"
#include "mapDistribute.H"

#include "AMIInterpolationF.H"
#include <thrust/iterator/counting_iterator.h>

// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //

template<class Type, class CombineOp>
void Foam::AMIInterpolation::interpolateToTarget
(
    const UList<Type>& fld,
    const CombineOp& cop,
    List<Type>& result,
    const UList<Type>& defaultValues
) const
{
    addProfiling(ami, "AMIInterpolation::interpolateToTarget");

    if (fld.size() != srcAddress_.size())
    {
        FatalErrorInFunction
            << "Supplied field size is not equal to source patch size" << nl
            << "    source patch   = " << srcAddress_.size() << nl
            << "    target patch   = " << tgtAddress_.size() << nl
            << "    supplied field = " << fld.size()
            << abort(FatalError);
    }

    if (lowWeightCorrection_ > 0)
    {
        if (defaultValues.size() != tgtAddress_.size())
        {
            FatalErrorInFunction
                << "Employing default values when sum of weights falls below "
                << lowWeightCorrection_
                << " but supplied default field size is not equal to target "
                << "patch size" << nl
                << "    default values = " << defaultValues.size() << nl
                << "    target patch   = " << tgtAddress_.size() << nl
                << abort(FatalError);
        }
    }

    result.setSize(tgtAddress_.size());

    if (distributed())
    {
        const mapDistribute& map = srcMapPtr_();

        List<Type> work(fld);
        map.distribute(work);

        forAll(result, facei)
        {
            if (tgtWeightsSum_[facei] < lowWeightCorrection_)
            {
                result[facei] = defaultValues[facei];
            }
            else
            {
                const labelList& faces = tgtAddress_[facei];
                const scalarList& weights = tgtWeights_[facei];

                forAll(faces, i)
                {
                    cop(result[facei], facei, work[faces[i]], weights[i]);
                }
            }
        }
    }
    else
    {
        forAll(result, facei)
        {
            if (tgtWeightsSum_[facei] < lowWeightCorrection_)
            {
                result[facei] = defaultValues[facei];
            }
            else
            {
                const labelList& faces = tgtAddress_[facei];
                const scalarList& weights = tgtWeights_[facei];

                forAll(faces, i)
                {
                    cop(result[facei], facei, fld[faces[i]], weights[i]);
                }
            }
        }
    }
}


template<class Type, class CombineOp>
void Foam::AMIInterpolation::interpolateToSource
(
    const UList<Type>& fld,
    const CombineOp& cop,
    List<Type>& result,
    const UList<Type>& defaultValues
) const
{
    addProfiling(ami, "AMIInterpolation::interpolateToSource");

    if (fld.size() != tgtAddress_.size())
    {
        FatalErrorInFunction
            << "Supplied field size is not equal to target patch size" << nl
            << "    source patch   = " << srcAddress_.size() << nl
            << "    target patch   = " << tgtAddress_.size() << nl
            << "    supplied field = " << fld.size()
            << abort(FatalError);
    }

    if (lowWeightCorrection_ > 0)
    {
        if (defaultValues.size() != srcAddress_.size())
        {
            FatalErrorInFunction
                << "Employing default values when sum of weights falls below "
                << lowWeightCorrection_
                << " but supplied default field size is not equal to source "
                << "patch size" << nl
                << "    default values = " << defaultValues.size() << nl
                << "    source patch   = " << srcAddress_.size() << nl
                << abort(FatalError);
        }
    }

    result.setSize(srcAddress_.size());

    if (distributed())
    {
        const mapDistribute& map = tgtMapPtr_();

        List<Type> work(fld);
        map.distribute(work);

        forAll(result, facei)
        {
            if (srcWeightsSum_[facei] < lowWeightCorrection_)
            {
                result[facei] = defaultValues[facei];
            }
            else
            {
                const labelList& faces = srcAddress_[facei];
                const scalarList& weights = srcWeights_[facei];

                forAll(faces, i)
                {
                    cop(result[facei], facei, work[faces[i]], weights[i]);
                }
            }
        }
    }
    else
    {
        forAll(result, facei)
        {
            if (srcWeightsSum_[facei] < lowWeightCorrection_)
            {
                result[facei] = defaultValues[facei];
            }
            else
            {
                const labelList& faces = srcAddress_[facei];
                const scalarList& weights = srcWeights_[facei];

                forAll(faces, i)
                {
                    cop(result[facei], facei, fld[faces[i]], weights[i]);
                }
            }
        }
    }
}


template<class Type, class CombineOp>
Foam::tmp<Foam::Field<Type>> Foam::AMIInterpolation::interpolateToSource
(
    const Field<Type>& fld,
    const CombineOp& cop,
    const UList<Type>& defaultValues
) const
{
    auto tresult = tmp<Field<Type>>::New(srcAddress_.size(), Zero);

    interpolateToSource
    (
        fld,
        multiplyWeightedOp<Type, CombineOp>(cop),
        tresult.ref(),
        defaultValues
    );

    return tresult;
}


template<class Type, class CombineOp>
Foam::tmp<Foam::Field<Type>> Foam::AMIInterpolation::interpolateToSource
(
    const tmp<Field<Type>>& tFld,
    const CombineOp& cop,
    const UList<Type>& defaultValues
) const
{
    return interpolateToSource(tFld(), cop, defaultValues);
}


template<class Type, class CombineOp>
Foam::tmp<Foam::Field<Type>> Foam::AMIInterpolation::interpolateToTarget
(
    const Field<Type>& fld,
    const CombineOp& cop,
    const UList<Type>& defaultValues
) const
{
    auto tresult = tmp<Field<Type>>::New(tgtAddress_.size(), Zero);

    interpolateToTarget
    (
        fld,
        multiplyWeightedOp<Type, CombineOp>(cop),
        tresult.ref(),
        defaultValues
    );

    return tresult;
}


template<class Type, class CombineOp>
Foam::tmp<Foam::Field<Type>> Foam::AMIInterpolation::interpolateToTarget
(
    const tmp<Field<Type>>& tFld,
    const CombineOp& cop,
    const UList<Type>& defaultValues
) const
{
    return interpolateToTarget(tFld(), cop, defaultValues);
}


template<class Type>
Foam::tmp<Foam::Field<Type>> Foam::AMIInterpolation::interpolateToSource
(
    const Field<Type>& fld,
    const UList<Type>& defaultValues
) const
{
    return interpolateToSource(fld, plusEqOp<Type>(), defaultValues);
}


template<class Type>
Foam::tmp<Foam::Field<Type>> Foam::AMIInterpolation::interpolateToSource
(
    const tmp<Field<Type>>& tFld,
    const UList<Type>& defaultValues
) const
{
    return interpolateToSource(tFld(), plusEqOp<Type>(), defaultValues);
}


template<class Type>
Foam::tmp<Foam::Field<Type>> Foam::AMIInterpolation::interpolateToTarget
(
    const Field<Type>& fld,
    const UList<Type>& defaultValues
) const
{
    return interpolateToTarget(fld, plusEqOp<Type>(), defaultValues);
}


template<class Type>
Foam::tmp<Foam::Field<Type>> Foam::AMIInterpolation::interpolateToTarget
(
    const tmp<Field<Type>>& tFld,
    const UList<Type>& defaultValues
) const
{
    return interpolateToTarget(tFld(), plusEqOp<Type>(), defaultValues);
}

// ************************************************************************* //

template<class Type, class CombineOp>
void Foam::AMIInterpolation::interpolateToSource
(
    const gpuList<Type>& fld,
    const CombineOp& cop,
    gpuList<Type>& result,
    const gpuList<Type>& defaultValues
) const
{
    if (fld.size() != tgtAddress_.size())
    {
        FatalErrorIn
        (
            "AMIInterpolation::interpolateToSource"
            "("
                "const gpuList<Type>&, "
                "const CombineOp&, "
                "gpuList<Type>&, "
                "const gpuList<Type>&"
            ") const"
        )   << "Supplied field size is not equal to target patch size" << nl
            << "    source patch   = " << srcAddress_.size() << nl
            << "    target patch   = " << tgtAddress_.size() << nl
            << "    supplied field = " << fld.size()
            << abort(FatalError);
    }

    if (lowWeightCorrection_ > 0)
    {
        if (defaultValues.size() != srcAddress_.size())
        {
            FatalErrorIn
            (
                "AMIInterpolation::interpolateToSource"
                "("
                    "const UList<Type>&, "
                    "const CombineOp&, "
                    "List<Type>&, "
                    "const UList<Type>&"
                ") const"
            )   << "Employing default values when sum of weights falls below "
                << lowWeightCorrection_
                << " but supplied default field size is not equal to target "
                << "patch size" << nl
                << "    default values = " << defaultValues.size() << nl
                << "    source patch   = " << srcAddress_.size() << nl
                << abort(FatalError);
        }
    }

    result.setSize(srcAddress_.size());

    gpuList<Type> work;
    const gpuList<Type>* listPtr;

    if (singlePatchProc_ == -1)
    {
        const mapDistribute& map = tgtMapPtr_();

        List<Type> workTmp(fld.size());
        thrust::copy(fld.begin(),fld.end(),workTmp.begin());
        
        map.distribute(workTmp);
        work = workTmp;

        listPtr = &work;
    }
    else
    {
        listPtr = &fld;
    }
 
    const gpuList<Type>& f = *listPtr;

    if(lowWeightCorrection_ > 0)
    {
        thrust::transform
        (
            thrust::make_counting_iterator(0),
            thrust::make_counting_iterator(0)+result.size(),
            result.begin(),
            AMIInterpolationInterpolateFunctor<Type,CombineOp>
            (
                lowWeightCorrection_,
                cop,
                defaultValues.data(),
                f.data(),
                srcgpuAddress_.data(),
                srcgpuStartAddress_.data(),
                srcgpuWeights_.data(),
                srcgpuWeightsSum_.data()
            )
        );
    }
    else
    {
        thrust::transform
        (
            thrust::make_counting_iterator(0),
            thrust::make_counting_iterator(0)+result.size(),
            result.begin(),
            AMIInterpolationInterpolateNoCorrectionFunctor<Type,CombineOp>
            (
                cop,
                f.data(),
                srcgpuAddress_.data(),
                srcgpuStartAddress_.data(),
                srcgpuWeights_.data()
            )
        );
    }
}


template<class Type, class CombineOp>
void Foam::AMIInterpolation::interpolateToTarget
(
    const gpuList<Type>& fld,
    const CombineOp& cop,
    gpuList<Type>& result,
    const gpuList<Type>& defaultValues
) const
{
    if (fld.size() != srcAddress_.size())
    {
        FatalErrorIn
        (
            "AMIInterpolation::interpolateToTarget"
            "("
                "const gpuList<Type>&, "
                "const CombineOp&, "
                "gpuList<Type>&, "
                "const gpuList<Type>&"
            ") const"
        )   << "Supplied field size is not equal to source patch size" << nl
            << "    source patch   = " << srcgpuAddress_.size() << nl
            << "    target patch   = " << tgtgpuAddress_.size() << nl
            << "    supplied field = " << fld.size()
            << abort(FatalError);
    }

    if (lowWeightCorrection_ > 0)
    {
        if (defaultValues.size() != tgtAddress_.size())
        {
            FatalErrorIn
            (
                "AMIInterpolation::interpolateToTarget"
                "("
                    "const gpuList<Type>&, "
                    "const CombineOp&, "
                    "gpuList<Type>&, "
                    "const gpuList<Type>&"
                ") const"
            )   << "Employing default values when sum of weights falls below "
                << lowWeightCorrection_
                << " but supplied default field size is not equal to target "
                << "patch size" << nl
                << "    default values = " << defaultValues.size() << nl
                << "    target patch   = " << tgtAddress_.size() << nl
                << abort(FatalError);
        }
    }

    result.setSize(tgtAddress_.size());

    gpuList<Type> work;
    const gpuList<Type>* listPtr;

    if (singlePatchProc_ == -1)
    {
        const mapDistribute& map = srcMapPtr_();

        List<Type> workTmp(fld.size());

        thrust::copy(fld.begin(),fld.end(),workTmp.begin());
        
        map.distribute(workTmp);
        work = workTmp;

        listPtr = &work;
    }
    else
    {
        listPtr = &fld;
    }
    
    const gpuList<Type>& f = *listPtr;

    if(lowWeightCorrection_ > 0)
    {
        thrust::transform
        (
            thrust::make_counting_iterator(0),
            thrust::make_counting_iterator(0)+result.size(),
            result.begin(),
            AMIInterpolationInterpolateFunctor<Type,CombineOp>
            (
                lowWeightCorrection_,
                cop,
                defaultValues.data(),
                f.data(),
                tgtgpuAddress_.data(),
                tgtgpuStartAddress_.data(),
                tgtgpuWeights_.data(),
                tgtgpuWeightsSum_.data()
            )
        );
    }
    else
    {
        thrust::transform
        (
            thrust::make_counting_iterator(0),
            thrust::make_counting_iterator(0)+result.size(),
            result.begin(),
            AMIInterpolationInterpolateNoCorrectionFunctor<Type,CombineOp>
            (
                cop,
                f.data(),
                tgtgpuAddress_.data(),
                tgtgpuStartAddress_.data(),
                tgtgpuWeights_.data()
            )
        );
    }
}


template<class Type, class CombineOp>
Foam::tmp<Foam::gpuField<Type>> Foam::AMIInterpolation::interpolateToSource
(
    const gpuField<Type>& fld,
    const CombineOp& cop,
    const gpuList<Type>& defaultValues
) const
{
    tmp<gpuField<Type> > tresult
    (
        new gpuField<Type>
        (
            srcAddress_.size(),
            pTraits<Type>::zero
        )
    );

    interpolateToSource
    (
        fld,
        multiplyWeightedOp<Type, CombineOp>(cop),
        tresult.ref(),
        defaultValues
    );

    return tresult;
}


template<class Type, class CombineOp>
Foam::tmp<Foam::gpuField<Type>> Foam::AMIInterpolation::interpolateToSource
(
    const tmp<gpuField<Type> >& tFld,
    const CombineOp& cop,
    const gpuList<Type>& defaultValues
) const
{
    return interpolateToSource(tFld.ref(), cop, defaultValues);
}


template<class Type, class CombineOp>
Foam::tmp<Foam::gpuField<Type>> Foam::AMIInterpolation::interpolateToTarget
(
    const gpuField<Type>& fld,
    const CombineOp& cop,
    const gpuList<Type>& defaultValues
) const
{
    tmp<gpuField<Type> > tresult
    (
        new gpuField<Type>
        (
            tgtAddress_.size(),
            pTraits<Type>::zero
        )
    );

    interpolateToTarget
    (
        fld,
        multiplyWeightedOp<Type, CombineOp>(cop),
        tresult.ref(),
        defaultValues
    );

    return tresult;
}


template<class Type, class CombineOp>
Foam::tmp<Foam::gpuField<Type>> Foam::AMIInterpolation::interpolateToTarget
(
    const tmp<gpuField<Type> >& tFld,
    const CombineOp& cop,
    const gpuList<Type>& defaultValues
) const
{
    return interpolateToTarget(tFld.ref(), cop, defaultValues);
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>> Foam::AMIInterpolation::interpolateToSource
(
    const gpuField<Type>& fld,
    const gpuList<Type>& defaultValues
) const
{
    return interpolateToSource(fld, plusEqOp<Type>(), defaultValues);
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>> Foam::AMIInterpolation::interpolateToSource
(
    const tmp<gpuField<Type> >& tFld,
    const gpuList<Type>& defaultValues
) const
{
    return interpolateToSource(tFld.ref(), plusEqOp<Type>(), defaultValues);
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>> Foam::AMIInterpolation::interpolateToTarget
(
    const gpuField<Type>& fld,
    const gpuList<Type>& defaultValues
) const
{
    return interpolateToTarget(fld, plusEqOp<Type>(), defaultValues);
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>> Foam::AMIInterpolation::interpolateToTarget
(
    const tmp<gpuField<Type> >& tFld,
    const gpuList<Type>& defaultValues
) const
{
    return interpolateToTarget(tFld.ref(), plusEqOp<Type>(), defaultValues);
}


// ************************************************************************* //
