/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2013 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::distributionModels::multiNormal

Description
    Particle-size distribution model wherein random samples are drawn
    from a mixture of a finite set of doubly-truncated univariate normal
    probability density functions:

    \f[
        g (\mathbf{x}; \mathbf{\mu}, \mathbf{\sigma}, A, B) =
            \sum_i w_i f(x_i; \mu_i, \sigma_i, A, B)
    \f]
    with for any distribution:

    \f[
        f(x; \mu, \sigma, A, B) =
            \frac{1}{\sigma}
            \frac{
                \phi \left( \frac{x - \mu}{\sigma} \right)
            }{
                \Phi \left( \frac{B - \mu}{\sigma} \right)
              - \Phi \left( \frac{A - \mu}{\sigma} \right)
            }
    \f]
    where

    \vartable
      f(x; \mu, \sigma, A, B) | Doubly-truncated univariate normal distribution
      \mu         | Mean of the parent general normal distribution
      \sigma      | Standard deviation of the parent general normal distribution
      \phi(\cdot) | General normal probability density function
      \Phi(\cdot) | General normal cumulative distribution function
      x           | Sample
      A           | Minimum of the distribution (the same for each distribution)
      B           | Maximum of the distribution (the same for each distribution)
      w_i         | Weighting factor
    \endvartable

    Constraints:
    - \f$ \infty > B > A > 0 \f$
    - \f$ x \in [B,A] \f$
    - \f$ \sigma^2 > 0 \f$
    - \f$ w_i >= 0 \f$

    Random samples are generated by a combination of the inverse transform
    sampling technique and categorical sampling in three steps:
    - Draw a sample from the uniform probability density function
    on the unit interval \f$u = (0, 1)\f$
    - Find the interval among normalised cumulative weight intervals
    wherein \f$ u \f$ resides
    - Draw a sample from the distribution corresponding to the interval by
    using the quantile function of the doubly-truncated univariate normal
    probability density function by the following expressions (similar to
    \c distributionModels::normal):

    \f[
        x = \mu + \sigma \sqrt{2} \, {erf}^{-1} \left( 2 p - 1 \right)
    \f]
    with

    \f[
        p = u \,
            \left(
                \Phi\left(
                    \frac{B - \mu}{\sigma}
                    \right)
              - \Phi\left(
                    \frac{A - \mu}{\sigma}
                    \right)
            \right)
          + \Phi\left( \frac{A - \mu}{\sigma} \right)
    \f]

    \f[
        \Phi(\xi) =
            \frac{1}{2}
            \left(
                1 + {erf}\left(\frac{\xi - \mu}{\sigma \sqrt{2} }\right)
            \right)
    \f]
    where \f$ u \f$ is another sample drawn from the uniform probability
    density function on the unit interval \f$ (0, 1) \f$.

Usage
    Minimal example by using \c constant/\<CloudProperties\>:
    \verbatim
    subModels
    {
        injectionModels
        {
            <name>
            {
                ...

                sizeDistribution
                {
                    type        multiNormal;
                    multiNormalDistribution
                    {
                        minValue  <min>;
                        maxValue  <max>;
                        mu
                        (
                            <mean1>
                            <mean2>
                            ...
                        );
                        sigma
                        (
                            <standard deviation1>
                            <standard deviation2>
                            ...
                        );
                        weight
                        (
                            <weight1>
                            <weight2>
                            ...
                        );
                    }
                }
            }
        }
    }
    \endverbatim

    where the entries mean:
    \table
      Property   | Description                       | Type | Reqd   | Deflt
      type       | Type name: multiNormal            | word | yes    | -
      multiNormalDistribution | Distribution settings  | dict | yes  | -
      minValue   | Minimum of the distribution       | scalar | yes  | -
      maxValue   | Maximum of the distribution       | scalar | yes  | -
      mu         | List of means of parent general normal <!--
                 --> distributions                | scalarList | yes | -
      sigma      | List of standard deviations of parent  <!--
                 --> general normal distributions | scalarList | yes | -
      weight   | List of weights of a given distribution in <!--
                 --> the distribution mixture     | scalarList | yes | -
    \endtable

Notes
  - The sum of normal distributions (i.e. a mixture distribution) should not
  be confused with the sum of normally-distributed random variables.
  - \c minValue and \c maxValue are the same for all distributions
  in the distribution mixture.
  - \c weight should always be input in a non-decreasing (i.e. monotonic) order.

SourceFiles
    multiNormal.C

\*---------------------------------------------------------------------------*/

#ifndef distributionModels_multiNormal_H
#define distributionModels_multiNormal_H

#include "distributionModel.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace distributionModels
{

/*---------------------------------------------------------------------------*\
                         Class multiNormal Declaration
\*---------------------------------------------------------------------------*/

class multiNormal
:
    public distributionModel
{
    // Private Data

        //- List of means of the parent general normal distributions
        List<scalar> mu_;

        //- List of standard deviations of
        //- the parent general normal distributions
        List<scalar> sigma_;

        //- List of weights of a given distribution in the mixture
        List<scalar> weight_;


public:

    //- Runtime type information
    TypeName("multiNormal");


    // Constructors

        //- Construct from components
        multiNormal(const dictionary& dict, Random& rndGen);

        //- Copy construct
        multiNormal(const multiNormal& p);

        //- Construct and return a clone
        virtual autoPtr<distributionModel> clone() const
        {
            return autoPtr<distributionModel>(new multiNormal(*this));
        }

        //- No copy assignment
        void operator=(const multiNormal&) = delete;


    //- Destructor
    virtual ~multiNormal() = default;


    // Member Functions

        //- Sample the distribution
        virtual scalar sample() const;

        //- Sample the normal distribution
        scalar sample(const scalar mu, const scalar sigma) const;

        //- Return the theoretical mean of the distribution
        virtual scalar meanValue() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace distributionModels
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
