/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2016 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::distributionModels::massRosinRammler

Description
    Particle-size distribution model wherein random samples are drawn
    from the two-parameter Rosin-Rammler (Weibull) probability density
    function corrected to take into account varying number of particles
    per parcels for fixed-mass parcels.

    "There is a factor of \f$x^3\f$ difference between the size distribution
    probability density function of individual particles and modeled parcels
    of particles, \f$ f_{parcels}(D) = x^3 f_{particles}(D) \f$, because the
    submodel parameter, \f$ PPP \f$ (number of particles per parcel) is based
    on a fixed mass per parcel which weights the droplet distribution by
    a factor proportional to \f$ 1/x^3 \f$ (i.e. \f$ PPP = \dot{m}_{total}
    \Delta_{t_{inj(per-parcel)}} / {m}_{particle}  \f$)." (YHD:p. 1374-1375).

    \c massRosinRammler should be preferred over \c RosinRammler
    when \c parcelBasisType is based on \c mass:
    \verbatim
        parcelBasisType mass;
    \endverbatim

    The doubly-truncated mass-corrected
    Rosin-Rammler probability density function:

    \f[
        f(x; \lambda, n, A, B) =
                x^3
                \frac{n}{\lambda}
                \left( \frac{x}{\lambda} \right)^{n-1}
                \frac{
                    \exp\{ -(\frac{x}{\lambda})^n \}
                }
                {
                    \exp\{ -(\frac{A}{\lambda})^n \}
                  - \exp\{ -(\frac{B}{\lambda})^n \}
                }
    \f]
    where

    \vartable
        f(x; \lambda, n, A, B) | Rosin-Rammler probability density function
        \lambda   | Scale parameter
        n         | Shape parameter
        x         | Sample
        A         | Minimum of the distribution
        B         | Maximum of the distribution
    \endvartable

    Constraints:
    - \f$ \lambda > 0 \f$
    - \f$ n > 0 \f$

    Random samples are generated by the inverse transform sampling technique
    by using the quantile function of the doubly-truncated two-parameter
    mass-corrected Rosin-Rammler (Weibull) probability density function:

    \f[
        d = \lambda \, Q(a, u)^{1/n}
    \f]
    with

    \f[
        a = \frac{3}{n} + 1
    \f]
    where \f$ Q(z_1, z_2) \f$ is the inverse of regularised lower incomplete
    gamma function, and \f$ u \f$ is a sample drawn from the uniform
    probability density function on the interval \f$ (x, y) \f$:

    \f[
        x = \gamma \left( a, \frac{A}{\lambda}^n \right)
    \f]

    \f[
        y = \gamma \left( a, \frac{B}{\lambda}^n \right)
    \f]
    where \f$ \gamma(z_1, z_2) \f$ is the lower incomplete gamma function.

    Reference:
    \verbatim
        Standard model (tag:YHD):
            Yoon, S. S., Hewson, J. C., DesJardin, P. E.,
            Glaze, D. J., Black, A. R., & Skaggs, R. R. (2004).
            Numerical modeling and experimental measurements of a high speed
            solid-cone water spray for use in fire suppression applications.
            International Journal of Multiphase Flow, 30(11), 1369-1388.
            DOI:10.1016/j.ijmultiphaseflow.2004.07.006
    \endverbatim

Usage
    Minimal example by using \c constant/\<CloudProperties\>:
    \verbatim
    subModels
    {
        injectionModels
        {
            <name>
            {
                ...
                parcelBasisType  mass;
                ...

                sizeDistribution
                {
                    type        massRosinRammler;
                    massRosinRammlerDistribution
                    {
                        lambda      <scaleParameterValue>;
                        n           <shapeParameterValue>;
                        minValue    <minValue>;
                        maxValue    <maxValue>;
                    }
                }
            }
        }
    }
    \endverbatim

    where the entries mean:
    \table
      Property     | Description                       | Type | Reqd | Deflt
      type         | Type name: massRosinRammler       | word | yes  | -
      massRosinRammlerDistribution | Distribution settings | dict | yes  | -
      lambda       | Scale parameter                   | scalar | yes  | -
      n            | Shape parameter                   | scalar | yes  | -
      minValue     | Minimum of the distribution       | scalar | yes  | -
      maxValue     | Maximum of the distribution       | scalar | yes  | -
    \endtable

Note
  - In the current context, \c lambda (or \c d) is called "characteristic
    droplet size", and \c n "empirical dimensionless constant to specify
    the distribution width, sometimes referred to as the dispersion
    coefficient." (YHD:p. 1374).

SourceFiles
    massRosinRammler.C

\*---------------------------------------------------------------------------*/

#ifndef massRosinRammler_H
#define massRosinRammler_H

#include "distributionModel.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace distributionModels
{

/*---------------------------------------------------------------------------*\
                       Class massRosinRammler Declaration
\*---------------------------------------------------------------------------*/

class massRosinRammler
:
    public distributionModel
{
    // Private Data

        //- Scale parameter
        scalar lambda_;

        //- Shape parameter
        scalar n_;


public:

    //- Runtime type information
    TypeName("massRosinRammler");


    // Constructors

        //- Construct from components
        massRosinRammler(const dictionary& dict, Random& rndGen);

        //- Copy construct
        massRosinRammler(const massRosinRammler& p);

        //- Construct and return a clone
        virtual autoPtr<distributionModel> clone() const
        {
            return autoPtr<distributionModel>(new massRosinRammler(*this));
        }

        //- No copy assignment
        void operator=(const massRosinRammler&) = delete;


    //- Destructor
    virtual ~massRosinRammler() = default;


    // Member Functions

        //- Sample the distribution
        virtual scalar sample() const;

        //- Return the theoretical mean of the distribution
        virtual scalar meanValue() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace distributionModels
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
