/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "cyclicgpuFvPatch.H"
#include "addToRunTimeSelectionTable.H"
#include "gpufvMesh.H"
#include "transform.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    defineTypeNameAndDebug(cyclicgpuFvPatch, 0);
    addToRunTimeSelectionTable(gpufvPatch, cyclicgpuFvPatch, polyPatch);

    struct cyclicMakeWeightsFunctor : public std::binary_function<scalar,scalar,scalar>
    {
         __host__ __device__
         scalar operator()(const scalar& di,const scalar& dni)
         {
             return dni/(di + dni);
         }
    };
    
    struct cyclicDeltaFunctor : public std::binary_function<vector,vector,vector>
    {
        const tensor t;
        cyclicDeltaFunctor(const tensor _t):t(_t){}
        __host__ __device__
        vector operator()(const vector& ddi, const vector& dni)
        {
            return ddi - transform(t, dni);
        }
    };
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::cyclicgpuFvPatch::makeWeights(scalargpuField& w) const
{
    const cyclicgpuFvPatch& nbrPatch = neighbFvPatch();

    const scalargpuField deltas(nf()&coupledgpuFvPatch::delta());
    const scalargpuField nbrDeltas(nbrPatch.nf()&nbrPatch.coupledgpuFvPatch::delta());
/*
    forAll(deltas, facei)
    {
        scalar di = deltas[facei];
        scalar dni = nbrDeltas[facei];

        w[facei] = dni/(di + dni);
    }
*/
    thrust::transform(deltas.begin(),deltas.end(),nbrDeltas.begin(),w.begin(),
                      cyclicMakeWeightsFunctor());
}


Foam::tmp<Foam::vectorgpuField> Foam::cyclicgpuFvPatch::delta() const
{
    const vectorgpuField patchD(coupledgpuFvPatch::delta());
    const vectorgpuField nbrPatchD(neighbFvPatch().coupledgpuFvPatch::delta());

    tmp<vectorgpuField> tpdv(new vectorgpuField(patchD.size()));
    vectorgpuField& pdv = tpdv.ref();

    // To the transformation if necessary
    if (parallel())
    {
/*
        forAll(patchD, facei)
        {
            vector ddi = patchD[facei];
            vector dni = nbrPatchD[facei];

            pdv[facei] = ddi - dni;
        }
*/
        thrust::transform(patchD.begin(),patchD.end(),nbrPatchD.begin(),pdv.begin(),
                          subtractOperatorFunctor<vector,vector,vector>());
    }
    else
    {
/*
        forAll(patchD, facei)
        {
            vector ddi = patchD[facei];
            vector dni = nbrPatchD[facei];

            pdv[facei] = ddi - transform(forwardT()[0], dni);
        }
*/
        thrust::transform(patchD.begin(),patchD.end(),nbrPatchD.begin(),pdv.begin(),
                          cyclicDeltaFunctor(gpuForwardT().first()));
    }

    return tpdv;
}


Foam::tmp<Foam::labelgpuField> Foam::cyclicgpuFvPatch::interfaceInternalField
(
    const labelgpuList& internalData
) const
{
    return patchInternalField(internalData);
}


Foam::tmp<Foam::labelgpuField> Foam::cyclicgpuFvPatch::interfaceInternalField
(
    const labelgpuList& internalData,
    const labelgpuList& faceCells
) const
{
    return patchInternalField(internalData, faceCells);
}


Foam::tmp<Foam::labelgpuField> Foam::cyclicgpuFvPatch::internalFieldTransfer
(
    const Pstream::commsTypes commsType,
    const labelgpuList& iF
) const
{
    return neighbFvPatch().patchInternalField(iF);
}

// ************************************************************************* //
