/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2016-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fvMesh

Description
    Mesh data needed to do the Finite Volume discretisation.

    NOTE ON USAGE:
    fvMesh contains all the topological and geometric information
    related to the mesh.  It is also responsible for keeping the data
    up-to-date.  This is done by deleting the cell volume, face area,
    cell/face centre, addressing and other derived information as
    required and recalculating it as necessary.  The fvMesh therefore
    reserves the right to delete the derived information upon every
    topological (mesh refinement/morphing) or geometric change (mesh
    motion).  It is therefore unsafe to keep local references to the
    derived data outside of the time loop.

SourceFiles
    fvMesh.C
    fvMeshGeometry.C

\*---------------------------------------------------------------------------*/

#ifndef gpufvMesh_H
#define gpufvMesh_H

#include "fvMesh.H"
#include "gpulduMesh.H"
#include "gpufvBoundaryMesh.H"
#include "gpusurfaceInterpolation.H"
#include "DimensionedgpuField.H"
#include "volgpuFieldsFwd.H"
#include "surfacegpuFieldsFwd.H"
#include "slicedVolgpuFieldsFwd.H"
#include "slicedSurfacegpuFieldsFwd.H"



// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

class gpufvMeshLduAddressing;
class gpuvolMesh;
template<class Type>
class gpufvMatrix;

/*---------------------------------------------------------------------------*\
                           Class fvMesh Declaration
\*---------------------------------------------------------------------------*/

class gpufvMesh
:
    public gpulduMesh,
    public gpusurfaceInterpolation    // needs input from fvSchemes
{
protected:

    // Private data
    
        //- host mesh
        fvMesh& hostmesh_;

        //- Boundary mesh
        gpufvBoundaryMesh boundary_;

    // Demand-driven data

        mutable gpufvMeshLduAddressing* lduPtr_;

        //- Current time index for cell volumes
        //  Note.  The whole mechanism will be replaced once the
        //  dimensionedField is created and the dimensionedField
        //  will take care of the old-time levels.
        mutable label curTimeIndex_;

        //- Cell volumes old time level
        mutable void* VPtr_;

        //- Cell volumes old time level
        mutable DimensionedgpuField<scalar, gpuvolMesh>* V0Ptr_;

        //- Cell volumes old-old time level
        mutable DimensionedgpuField<scalar, gpuvolMesh>* V00Ptr_;

        //- Face area vectors
        mutable slicedSurfaceVectorgpuField* SfPtr_;

        //- Mag face area vectors
        mutable surfaceScalargpuField* magSfPtr_;

        //- Cell centres
        mutable slicedVolVectorgpuField* CPtr_;

        //- Face centres
        mutable slicedSurfaceVectorgpuField* CfPtr_;

        //- Face motion fluxes
        mutable surfaceScalargpuField* phiPtr_;


       // Make geometric data

            void makeSf() const;
            void makeMagSf() const;

            void makeC() const;
            void makeCf() const;


        //- No copy construct
        gpufvMesh(const gpufvMesh&) = delete;

        //- No copy assignment
        void operator=(const gpufvMesh&) = delete;


public:

    // Public typedefs

        typedef gpufvMesh Mesh;
        typedef gpufvBoundaryMesh BoundaryMesh;


    // Declare name of the class and its debug switch
    ClassName("gpufvMesh");


    // Constructors

        //- Construct from fvMesh
        gpufvMesh(const fvMesh& baseMesh);


    //- Destructor
    virtual ~gpufvMesh();


    // Member Functions

	//- Add boundary patches. Constructor helper
	void addFvPatches
	(
		PtrList<polyPatch>& plist,
		const bool validBoundary = true
	);
	
	//- Add boundary patches. Constructor helper
	void addFvPatches
	(
		const List<polyPatch*>& p,
		const bool validBoundary = true
	);

        // Access

	        //- Return reference to name
	        inline fvMesh& hostmesh()      noexcept
    	    {
			    return hostmesh_;
			}
			
	        inline fvMesh& hostmesh() const      noexcept
    	    {
			    return hostmesh_;
			}

            //- Return the top-level database
            const Time& time() const
            {
                return hostmesh_.time();
            }

            //- Return true if thisDb() is a valid DB
            virtual bool hasDb() const
            {
                return hostmesh_.hasDb();
            }

            //- Return the object registry - resolve conflict polyMesh/lduMesh
            virtual const objectRegistry& thisDb() const
            {
                return hostmesh_.thisDb();
            }

            //- Return reference to name
            //  Note: name() is currently ambiguous due to derivation from
            //  surfaceInterpolation
            const word& name() const
            {
                return hostmesh_.name();
            }

            //- Return reference to boundary mesh
            const gpufvBoundaryMesh& boundary() const;

            //- Return ldu addressing
            virtual const gpulduAddressing& lduAddr() const;

            //- Return a list of pointers for each patch
            //  with only those pointing to interfaces being set
            virtual gpulduInterfacePtrsList interfaces() const;

            //- Return communicator used for parallel communication
            virtual label comm() const
            {
                return hostmesh_.comm();
            }


            // Overlap

/*                //- Interpolate interpolationCells only
                virtual void interpolate(volScalarField&) const
                {}

                //- Interpolate interpolationCells only
                virtual void interpolate(volVectorField&) const
                {}

                //- Interpolate interpolationCells only
                virtual void interpolate(volSphericalTensorField&) const
                {}

                //- Interpolate interpolationCells only
                virtual void interpolate(volSymmTensorField&) const
                {}

                //- Interpolate interpolationCells only
                virtual void interpolate(volTensorField&) const
                {}

                //- Interpolate interpolationCells only. No bcs.
                virtual void interpolate(scalarField&) const
                {}

                //- Interpolate interpolationCells only. No bcs.
                virtual void interpolate(vectorField&) const
                {}

                //- Interpolate interpolationCells only. No bcs.
                virtual void interpolate(sphericalTensorField&) const
                {}

                //- Interpolate interpolationCells only. No bcs.
                virtual void interpolate(symmTensorField&) const
                {}

                //- Interpolate interpolationCells only. No bcs.
                virtual void interpolate(tensorField&) const
                {}
*/
                //- Solve returning the solution statistics given convergence
                //- tolerance. Use the given solver controls
                virtual SolverPerformance<scalar> solve
                (
                    gpufvMatrix<scalar>&,
                    const dictionary&
                ) const;

                //- Solve returning the solution statistics given convergence
                //- tolerance. Use the given solver controls
                virtual SolverPerformance<vector> solve
                (
                    gpufvMatrix<vector>&,
                    const dictionary&
                ) const;

                //- Solve returning the solution statistics given convergence
                //- tolerance. Use the given solver controls
                virtual SolverPerformance<sphericalTensor> solve
                (
                    gpufvMatrix<sphericalTensor>&,
                    const dictionary&
                ) const;

                //- Solve returning the solution statistics given convergence
                //- tolerance. Use the given solver controls
                virtual SolverPerformance<symmTensor> solve
                (
                    gpufvMatrix<symmTensor>&,
                    const dictionary&
                ) const;

                //- Solve returning the solution statistics given convergence
                //- tolerance. Use the given solver controls
                virtual SolverPerformance<tensor> solve
                (
                    gpufvMatrix<tensor>&,
                    const dictionary&
                ) const;


            //- Internal face owner. Note bypassing virtual mechanism so
            //  e.g. relaxation always gets done using original addressing
            const labelgpuList& owner() const
            {
                return gpufvMesh::lduAddr().lowerAddr();
            }

            //- Internal face neighbour
            const labelgpuList& neighbour() const
            {
                return gpufvMesh::lduAddr().upperAddr();
            }

            //- Return cell volumes
            const DimensionedgpuField<scalar, gpuvolMesh>& V() const;

            //- Return old-time cell volumes
            const DimensionedgpuField<scalar, gpuvolMesh>& V0() const;

            //- Return old-old-time cell volumes
            const DimensionedgpuField<scalar, gpuvolMesh>& V00() const;

            //- Return sub-cycle cell volumes
            tmp<DimensionedgpuField<scalar, gpuvolMesh>> Vsc() const;

            //- Return sub-cycle old-time cell volumes
            tmp<DimensionedgpuField<scalar, gpuvolMesh>> Vsc0() const;

            //- Return cell face area vectors
            const surfaceVectorgpuField& Sf() const;

            //- Return cell face area magnitudes
            const surfaceScalargpuField& magSf() const;

            //- Return cell face motion fluxes
            const surfaceScalargpuField& phi() const;

            //- Return cell centres as volVectorField
            const volVectorgpuField& C() const;

            //- Return face centres as surfaceVectorField
            const surfaceVectorgpuField& Cf() const;

            //- Return face deltas as surfaceVectorField
            tmp<surfaceVectorgpuField> delta() const;


        // Edit

            //- Clear all geometry and addressing
            void clearOut();
		    void clearGeom();

            //- Update all geometric data. This gets redirected up from
            //- primitiveMesh level
            virtual void updateGeom();

            //- Remove boundary patches. Warning: fvPatchFields hold ref to
            //- these fvPatches.
            void removeFvBoundary();

            //- Return cell face motion fluxes
            surfaceScalargpuField& setPhi();

            //- Return old-time cell volumes
            DimensionedgpuField<scalar, gpuvolMesh>& setV0();




    // Member Operators

        //- Compares addresses
        bool operator!=(const gpufvMesh& rhs) const;

        //- Compares addresses
        bool operator==(const gpufvMesh& rhs) const;
};



// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
//    #include "gpufvMeshTemplates.C"
    #include "gpufvPatchFvMeshTemplates.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
