/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2018-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpufvBoundaryMesh.H"
#include "gpufvMesh.H"

// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //

void Foam::gpufvBoundaryMesh::addPatches(const polyBoundaryMesh& basicBdry)
{
    setSize(basicBdry.size());

    // Set boundary patches
    gpufvPatchList& Patches = *this;

    forAll(Patches, patchi)
    {
        Patches.set(patchi, gpufvPatch::New(basicBdry[patchi], *this));
    }
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::gpufvBoundaryMesh::gpufvBoundaryMesh
(
    const gpufvMesh& m
)
:
    gpufvPatchList(),
    mesh_(m)
{}


Foam::gpufvBoundaryMesh::gpufvBoundaryMesh
(
    const gpufvMesh& m,
    const polyBoundaryMesh& basicBdry
)
:
    gpufvPatchList(basicBdry.size()),
    mesh_(m)
{
    addPatches(basicBdry);
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

Foam::labelList Foam::gpufvBoundaryMesh::indices
(
    const wordRe& matcher,
    const bool useGroups
) const
{
    return mesh().hostmesh().boundaryMesh().indices(matcher, useGroups);
}


Foam::labelList Foam::gpufvBoundaryMesh::indices
(
    const wordRes& matcher,
    const bool useGroups
) const
{
    return mesh().hostmesh().boundaryMesh().indices(matcher, useGroups);
}


Foam::label Foam::gpufvBoundaryMesh::findPatchID(const word& patchName) const
{
    if (patchName.empty())
    {
        return -1;
    }

    const gpufvPatchList& patches = *this;

    forAll(patches, patchi)
    {
        if (patches[patchi].name() == patchName)
        {
            return patchi;
        }
    }

    // Not found, return -1
    return -1;
}


/*void Foam::gpufvBoundaryMesh::movePoints()
{
    gpufvPatchList& patches = *this;

    for (gpufvPatch& p : patches)
    {
        p.initMovePoints();
    }

    for (gpufvPatch& p : patches)
    {
        p.movePoints();
    }
}*/


Foam::UPtrList<const Foam::labelgpuList>
Foam::gpufvBoundaryMesh::gpuFaceCells() const
{
    const gpufvPatchList& patches = *this;

    UPtrList<const labelgpuList> list(patches.size());

    forAll(list, patchi)
    {
        list.set(patchi, &patches[patchi].gpuFaceCells());
    }

    return list;
}

Foam::gpulduInterfacePtrsList Foam::gpufvBoundaryMesh::interfaces() const
{
    const gpufvPatchList& patches = *this;

    gpulduInterfacePtrsList list(patches.size());

    forAll(list, patchi)
    {
        const gpulduInterface* lduPtr = isA<gpulduInterface>(patches[patchi]);

        if (lduPtr)
        {
            list.set(patchi, lduPtr);
        }
    }

    return list;
}


// * * * * * * * * * * * * Protected Member Functions  * * * * * * * * * * * //

/*void Foam::fvBoundaryMesh::readUpdate(const polyBoundaryMesh& basicBdry)
{
    clear();
    addPatches(basicBdry);
}*/


// * * * * * * * * * * * * * * Member Operators  * * * * * * * * * * * * * * //

const Foam::gpufvPatch& Foam::gpufvBoundaryMesh::operator[]
(
    const word& patchName
) const
{
    const label patchi = findPatchID(patchName);

    if (patchi < 0)
    {
        FatalErrorInFunction
            << "Patch named " << patchName << " not found." << nl
            << abort(FatalError);
    }

    return operator[](patchi);
}


Foam::gpufvPatch& Foam::gpufvBoundaryMesh::operator[]
(
    const word& patchName
)
{
    const label patchi = findPatchID(patchName);

    if (patchi < 0)
    {
        FatalErrorInFunction
            << "Patch named " << patchName << " not found." << nl
            << abort(FatalError);
    }

    return operator[](patchi);
}


// ************************************************************************* //
