/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpufvcDdt.H"
#include "gpufvMesh.H"
#include "gpuddtScheme.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fvc
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
ddt
(
    const dimensioned<Type> dt,
    const gpufvMesh& mesh
)
{
    return fv::gpuddtScheme<Type>::New
    (
        mesh,
        mesh.hostmesh().ddtScheme("ddt(" + dt.name() + ')')
    ).ref().fvcDdt(dt);
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
ddt
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    return fv::gpuddtScheme<Type>::New
    (
        vf.mesh(),
        vf.mesh().hostmesh().ddtScheme("ddt(" + vf.name() + ')')
    ).ref().fvcDdt(vf);
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
ddt
(
    const dimensionedScalar& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    return fv::gpuddtScheme<Type>::New
    (
        vf.mesh(),
        vf.mesh().hostmesh().ddtScheme("ddt(" + rho.name() + ',' + vf.name() + ')')
    ).ref().fvcDdt(rho, vf);
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
ddt
(
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    return fv::gpuddtScheme<Type>::New
    (
        vf.mesh(),
        vf.mesh().hostmesh().ddtScheme("ddt(" + rho.name() + ',' + vf.name() + ')')
    ).ref().fvcDdt(rho, vf);
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
ddt
(
    const volScalargpuField& alpha,
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    return fv::gpuddtScheme<Type>::New
    (
        vf.mesh(),
        vf.mesh().hostmesh().ddtScheme
        (
            "ddt("
          + alpha.name() + ','
          + rho.name() + ','
          + vf.name() + ')'
        )
    ).ref().fvcDdt(alpha, rho, vf);
}


template<class Type>
tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
ddt
(
    const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& sf
)
{
    return fv::gpuddtScheme<Type>::New
    (
        sf.mesh(),
        sf.mesh().hostmesh().ddtScheme("ddt(" + sf.name() + ')')
    ).ref().fvcDdt(sf);
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
ddt
(
    const one&,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
)
{
    return ddt(vf);
}


template<class Type>
tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>
ddt
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf,
    const one&
)
{
    return ddt(vf);
}


template<class Type>
tmp<GeometricgpuField<typename Foam::flux<Type>::type, fvsPatchgpuField, gpusurfaceMesh>>
ddtCorr
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
)
{
    return fv::gpuddtScheme<Type>::New
    (
        U.mesh(),
        U.mesh().hostmesh().ddtScheme("ddt(" + U.name() + ')')
    ).ref().fvcDdtUfCorr(U, Uf);
}


template<class Type>
tmp<GeometricgpuField<typename Foam::flux<Type>::type, fvsPatchgpuField, gpusurfaceMesh>>
ddtCorr
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField
    <
        typename Foam::flux<Type>::type,
        fvsPatchgpuField,
        gpusurfaceMesh
    >& phi
)
{
    return fv::gpuddtScheme<Type>::New
    (
        U.mesh(),
        U.mesh().hostmesh().ddtScheme("ddt(" + U.name() + ')')
    ).ref().fvcDdtPhiCorr(U, phi);
}


template<class Type>
tmp<GeometricgpuField<typename Foam::flux<Type>::type, fvsPatchgpuField, gpusurfaceMesh>>
ddtCorr
(
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField
    <
        typename Foam::flux<Type>::type,
        fvsPatchgpuField,
        gpusurfaceMesh
    >& phi,
    const autoPtr<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>& Uf
)
{
    if (U.mesh().hostmesh().dynamic())
    {
        return ddtCorr(U, Uf());
    }
    else
    {
        return ddtCorr(U, phi);
    }
}


template<class Type>
tmp<GeometricgpuField<typename Foam::flux<Type>::type, fvsPatchgpuField, gpusurfaceMesh>>
ddtCorr
(
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
)
{
    return fv::gpuddtScheme<Type>::New
    (
        U.mesh(),
        U.mesh().hostmesh().ddtScheme("ddt(" + U.name() + ')')
    ).ref().fvcDdtUfCorr(rho, U, Uf);
}


template<class Type>
tmp<GeometricgpuField<typename Foam::flux<Type>::type, fvsPatchgpuField, gpusurfaceMesh>>
ddtCorr
(
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField
    <
        typename Foam::flux<Type>::type,
        fvsPatchgpuField,
        gpusurfaceMesh
    >& phi
)
{
    return fv::gpuddtScheme<Type>::New
    (
        U.mesh(),
        U.mesh().hostmesh().ddtScheme("ddt(" + rho.name() + ',' + U.name() + ')')
    ).ref().fvcDdtPhiCorr(rho, U, phi);
}


template<class Type>
tmp<GeometricgpuField<typename Foam::flux<Type>::type, fvsPatchgpuField, gpusurfaceMesh>>
ddtCorr
(
    const volScalargpuField& rho,
    const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField
    <
        typename Foam::flux<Type>::type,
        fvsPatchgpuField,
        gpusurfaceMesh
    >& phi,
    const autoPtr<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>& Uf
)
{
    if (U.mesh().hostmesh().dynamic())
    {
        return ddtCorr(rho, U, Uf());
    }
    else
    {
        return ddtCorr(rho, U, phi);
    }
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fvc

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// ************************************************************************* //
