/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2013 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::cyclicGAMGInterfaceField

Description
    GAMG agglomerated cyclic interface field.

SourceFiles
    cyclicGAMGInterfaceField.C

\*---------------------------------------------------------------------------*/

#ifndef cyclicGGAMGInterfaceField_H
#define cyclicGGAMGInterfaceField_H

#include "GGAMGInterfaceField.H"
#include "cyclicGGAMGInterface.H"
#include "cyclicLduInterfacegpuField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                  Class cyclicGAMGInterfaceField Declaration
\*---------------------------------------------------------------------------*/

class cyclicGGAMGInterfaceField
:
    public GGAMGInterfaceField,
    virtual public cyclicLduInterfacegpuField
{
    // Private data

        //- Local reference cast into the cyclic interface
        const cyclicGGAMGInterface& cyclicInterface_;

        //- Is the transform required
        bool doTransform_;

        //- Rank of component for transformation
        int rank_;


    // Private Member Functions

        //- No copy construct
        cyclicGGAMGInterfaceField(const cyclicGGAMGInterfaceField&) = delete;

        //- No copy assignment
        void operator=(const cyclicGGAMGInterfaceField&) = delete;


public:

    //- Runtime type information
    TypeName("cyclic");


    // Constructors

        //- Construct from GAMG interface and fine level interface field
        cyclicGGAMGInterfaceField
        (
            const GGAMGInterface& GAMGCp,
            const lduInterfacegpuField& fineInterfaceField
        );

        //- Construct from GAMG interface and fine level interface field
        cyclicGGAMGInterfaceField
        (
            const GGAMGInterface& GAMGCp,
            const bool doTransform,
            const int rank
        );


    //- Destructor
    virtual ~cyclicGGAMGInterfaceField() = default;


    // Member Functions

        // Access

            //- Return size
            label size() const
            {
                return cyclicInterface_.size();
            }


        //- Cyclic interface functions

            //- Does the interface field perform the transformation
            virtual bool doTransform() const
            {
                return doTransform_;
            }

            //- Return face transformation tensor
            virtual const tensorField& forwardT() const
            {
                return cyclicInterface_.forwardT();
            }

            virtual const tensorgpuField& gpuForwardT() const
            {
                return cyclicInterface_.gpuForwardT();
            }

            //- Return neighbour-cell transformation tensor
            virtual const tensorField& reverseT() const
            {
                return cyclicInterface_.reverseT();
            }

            virtual const tensorgpuField& gpuReverseT() const
            {
                return cyclicInterface_.gpuReverseT();
            }

            //- Return rank of component for transform
            virtual int rank() const
            {
                return rank_;
            }


        // Interface matrix update

            //- Update result field based on interface functionality
            /*virtual void updateInterfaceMatrix
            (
                solveScalarField& result,
                const bool add,
                const gpulduAddressing& lduAddr,
                const label patchId,
                const solveScalarField& psiInternal,
                const scalarField& coeffs,
                const direction cmpt,
                const Pstream::commsTypes commsType
            ) const;*/

            virtual void updateInterfaceMatrix
            (
                scalargpuField& result,
                const bool add,
                const gpulduAddressing& lduAddr,
                const label patchId,
                const scalargpuField& psiInternal,
                const scalargpuField& coeffs,
                const direction cmpt,
                const Pstream::commsTypes commsType
            ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
