/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2019-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "gpulduMatrix.H"
#include "IOstreams.H"
#include "Switch.H"
#include "objectRegistry.H"
#include "scalarIOField.H"
#include "Time.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
    defineTypeNameAndDebug(gpulduMatrix, 1);
}


const Foam::label Foam::gpulduMatrix::solver::defaultMaxIter_ = 1000;


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

Foam::gpulduMatrix::gpulduMatrix(const gpulduMesh& mesh)
:
    lduMesh_(mesh),
    lowerPtr_(nullptr),
    diagPtr_(nullptr),
    upperPtr_(nullptr),
    gLowerSortPtr_(nullptr),
    gUpperSortPtr_(nullptr),
    coarsestLevel_(false)
{}


Foam::gpulduMatrix::gpulduMatrix(const gpulduMatrix& A)
:
    lduMesh_(A.lduMesh_),
    lowerPtr_(nullptr),
    diagPtr_(nullptr),
    upperPtr_(nullptr),
    gLowerSortPtr_(nullptr),
    gUpperSortPtr_(nullptr),
    coarsestLevel_(false)

{
    if (A.lowerPtr_)
    {
        lowerPtr_ = new scalargpuField(*(A.lowerPtr_));
    }

    if (A.diagPtr_)
    {
        diagPtr_ = new scalargpuField(*(A.diagPtr_));
    }

    if (A.upperPtr_)
    {
        upperPtr_ = new scalargpuField(*(A.upperPtr_));
    }

    if (A.gLowerSortPtr_)
    {
        gLowerSortPtr_ = new scalargpuField(*(A.gLowerSortPtr_));
    }

    if (A.gUpperSortPtr_)
    {
        gUpperSortPtr_ = new scalargpuField(*(A.gUpperSortPtr_));
    }
}


Foam::gpulduMatrix::gpulduMatrix(gpulduMatrix& A, bool reuse)
:
    lduMesh_(A.lduMesh_),
    lowerPtr_(nullptr),
    diagPtr_(nullptr),
    upperPtr_(nullptr),
    gLowerSortPtr_(nullptr),
    gUpperSortPtr_(nullptr),
    coarsestLevel_(false)
{
    if (reuse)
    {
        if (A.lowerPtr_)
        {
            lowerPtr_ = A.lowerPtr_;
            A.lowerPtr_ = nullptr;
        }

        if (A.diagPtr_)
        {
            diagPtr_ = A.diagPtr_;
            A.diagPtr_ = nullptr;
        }

        if (A.upperPtr_)
        {
            upperPtr_ = A.upperPtr_;
            A.upperPtr_ = nullptr;
        }
    }
    else
    {
        if (A.lowerPtr_)
        {
            lowerPtr_ = new scalargpuField(*(A.lowerPtr_));
        }

        if (A.diagPtr_)
        {
            diagPtr_ = new scalargpuField(*(A.diagPtr_));
        }

        if (A.upperPtr_)
        {
            upperPtr_ = new scalargpuField(*(A.upperPtr_));
        }
    }
}


Foam::gpulduMatrix::gpulduMatrix(const gpulduMesh& mesh, Istream& is)
:
    lduMesh_(mesh),
    lowerPtr_(nullptr),
    diagPtr_(nullptr),
    upperPtr_(nullptr),
    gLowerSortPtr_(nullptr),
    gUpperSortPtr_(nullptr),
    coarsestLevel_(false)
{
    Switch hasLow(is);
    Switch hasDiag(is);
    Switch hasUp(is);

    if (hasLow)
    {
        lowerPtr_ = new scalargpuField(is);    
    }
    if (hasDiag)
    {
        diagPtr_ = new scalargpuField(is);
    }
    if (hasUp)
    {
        upperPtr_ = new scalargpuField(is);
    }
}


Foam::gpulduMatrix::~gpulduMatrix()
{
    if (lowerPtr_)
    {
        delete lowerPtr_;
    }

    if (diagPtr_)
    {
        delete diagPtr_;
    }

    if (upperPtr_)
    {
        delete upperPtr_;
    }

    if (gLowerSortPtr_)
    {
        delete gLowerSortPtr_;
    }
 
    if (gUpperSortPtr_)
    {
        delete gUpperSortPtr_;
    }
}


Foam::scalargpuField& Foam::gpulduMatrix::gpuLower()
{
    if (!lowerPtr_)
    {
        if (upperPtr_)
        {
            lowerPtr_ = new scalargpuField(*upperPtr_);
        }
        else
        {
            lowerPtr_ = new scalargpuField(lduAddr().lowerAddr().size(), Zero);
        }
    }

    return *lowerPtr_;
}


Foam::scalargpuField& Foam::gpulduMatrix::gpuDiag()
{
    if (!diagPtr_)
    {
        diagPtr_ = new scalargpuField(lduAddr().size(), Zero);
    }

    return *diagPtr_;
}


Foam::scalargpuField& Foam::gpulduMatrix::gpuUpper()
{
    if (!upperPtr_)
    {
        if (lowerPtr_)
        {
            upperPtr_ = new scalargpuField(*lowerPtr_);
        }
        else
        {
            upperPtr_ = new scalargpuField(lduAddr().lowerAddr().size(), Zero);
        }
    }

    return *upperPtr_;
}


Foam::scalargpuField& Foam::gpulduMatrix::gpuLower(const label nCoeffs)
{
    if (!lowerPtr_)
    {
        if (upperPtr_)
        {
            lowerPtr_ = new scalargpuField(*upperPtr_);
        }
        else
        {
            lowerPtr_ = new scalargpuField(nCoeffs, Zero);
        }
    }

    return *lowerPtr_;
}


Foam::scalargpuField& Foam::gpulduMatrix::gpuDiag(const label size)
{
    if (!diagPtr_)
    {
        diagPtr_ = new scalargpuField(size, Zero);
    }

    return *diagPtr_;
}


Foam::scalargpuField& Foam::gpulduMatrix::gpuUpper(const label nCoeffs)
{
    if (!upperPtr_)
    {
        if (lowerPtr_)
        {
            upperPtr_ = new scalargpuField(*lowerPtr_);
        }
        else
        {
            upperPtr_ = new scalargpuField(nCoeffs, Zero);
        }
    }

    return *upperPtr_;
}


const Foam::scalargpuField& Foam::gpulduMatrix::gpuLower() const
{
    if (!lowerPtr_ && !upperPtr_)
    {
        FatalErrorInFunction
            << "lowerPtr_ or upperPtr_ unallocated"
            << abort(FatalError);
    }

    if (lowerPtr_)
    {
        return *lowerPtr_;
    }
    else
    {
        return *upperPtr_;
    }
}


const Foam::scalargpuField& Foam::gpulduMatrix::gpuDiag() const
{
    if (!diagPtr_)
    {
        FatalErrorInFunction
            << "diagPtr_ unallocated"
            << abort(FatalError);
    }

    return *diagPtr_;
}


const Foam::scalargpuField& Foam::gpulduMatrix::gpuUpper() const
{
    if (!lowerPtr_ && !upperPtr_)
    {
        FatalErrorInFunction
            << "lowerPtr_ or upperPtr_ unallocated"
            << abort(FatalError);
    }

    if (upperPtr_)
    {
        return *upperPtr_;
    }
    else
    {
        return *lowerPtr_;
    }
}

//************************************************************************

void Foam::gpulduMatrix::calcSortCoeffs
(
    scalargpuField& out, 
    const scalargpuField& in
) const
{
    const labelgpuList& sort = lduAddr().losortAddr();

    thrust::copy
    (
        thrust::make_permutation_iterator
        (
            in.begin(),
            sort.begin()
        ),
        thrust::make_permutation_iterator
        (
            in.begin(),
            sort.end()
        ),
        out.begin()
    );
}

const Foam::scalargpuField& Foam::gpulduMatrix::gpuLowerSort() const
{
    if (!lowerPtr_ && !upperPtr_)
    {
    	FatalErrorInFunction
			<< "lowerPtr_ or upperPtr_ unallocated"
            << abort(FatalError);
    }

    if(gLowerSortPtr_)
    {
        return *gLowerSortPtr_;
    }

    if (lowerPtr_)
    {   
        gLowerSortPtr_ = new scalargpuField(gpuLower().size());

        calcSortCoeffs(*gLowerSortPtr_,*lowerPtr_);
      
        return *gLowerSortPtr_;
    }
    else
    {
        if(!gUpperSortPtr_)
        {
            gUpperSortPtr_ = new scalargpuField(gpuUpper().size());

            calcSortCoeffs(*gUpperSortPtr_,*upperPtr_);
        }
      
        return *gUpperSortPtr_;
    }
}

const Foam::scalargpuField& Foam::gpulduMatrix::gpuUpperSort() const
{
    if (!lowerPtr_ && !upperPtr_)
    {
        FatalErrorInFunction
            << "lowerPtr_ or upperPtr_ unallocated"
            << abort(FatalError);
    }

    if(gUpperSortPtr_)
    {
        return *gUpperSortPtr_;
    }

    if (upperPtr_)
    {
        gUpperSortPtr_ = new scalargpuField(gpuUpper().size());

        calcSortCoeffs(*gUpperSortPtr_,*upperPtr_);
      
        return *gUpperSortPtr_;
    }
    else
    {
        if(!gLowerSortPtr_)
        {
            gLowerSortPtr_ = new scalargpuField(gpuLower().size());

            calcSortCoeffs(*gLowerSortPtr_,*lowerPtr_);
        }
      
        return *gLowerSortPtr_;
    }
}

//***************************************************************************


void Foam::gpulduMatrix::setResidualField
(
    const scalarField& residual,
    const word& fieldName,
    const bool initial
) const
{
    if (!mesh().hasDb())
    {
        return;
    }

    scalarIOField* residualPtr =
        mesh().thisDb().getObjectPtr<scalarIOField>
        (
            initial
          ? IOobject::scopedName("initialResidual", fieldName)
          : IOobject::scopedName("residual", fieldName)
        );

    if (residualPtr)
    {
        const IOdictionary* dataPtr =
            mesh().thisDb().findObject<IOdictionary>("data");

        if (dataPtr)
        {
            if (initial && dataPtr->found("firstIteration"))
            {
                *residualPtr = residual;
                DebugInfo
                    << "Setting residual field for first solver iteration "
                    << "for solver field: " << fieldName << endl;
            }
        }
        else
        {
            *residualPtr = residual;
            DebugInfo
                << "Setting residual field for solver field "
                << fieldName << endl;
        }
    }
}


// * * * * * * * * * * * * * * * Friend Operators  * * * * * * * * * * * * * //

Foam::Ostream& Foam::operator<<(Ostream& os, const gpulduMatrix& ldum)
{
    Switch hasLow = ldum.hasLower();
    Switch hasDiag = ldum.hasDiag();
    Switch hasUp = ldum.hasUpper();

    os  << hasLow << token::SPACE << hasDiag << token::SPACE
        << hasUp << token::SPACE;

    if (hasLow)
    {
        os  << ldum.gpuLower();
    }

    if (hasDiag)
    {
        os  << ldum.gpuDiag();
    }

    if (hasUp)
    {
        os  << ldum.gpuUpper();
    }

    os.check(FUNCTION_NAME);

    return os;
}


Foam::Ostream& Foam::operator<<(Ostream& os, const InfoProxy<gpulduMatrix>& ip)
{
    const gpulduMatrix& ldum = ip.t_;

    Switch hasLow = ldum.hasLower();
    Switch hasDiag = ldum.hasDiag();
    Switch hasUp = ldum.hasUpper();

    os  << "Lower:" << hasLow
        << " Diag:" << hasDiag
        << " Upper:" << hasUp << endl;

    if (hasLow)
    {
        os  << "lower:" << ldum.gpuLower().size() << endl;
    }
    if (hasDiag)
    {
        os  << "diag :" << ldum.gpuDiag().size() << endl;
    }
    if (hasUp)
    {
        os  << "upper:" << ldum.gpuUpper().size() << endl;
    }


    //if (hasLow)
    //{
    //    os  << "lower contents:" << endl;
    //    forAll(ldum.lower(), i)
    //    {
    //        os  << "i:" << i << "\t" << ldum.lower()[i] << endl;
    //    }
    //    os  << endl;
    //}
    //if (hasDiag)
    //{
    //    os  << "diag contents:" << endl;
    //    forAll(ldum.diag(), i)
    //    {
    //        os  << "i:" << i << "\t" << ldum.diag()[i] << endl;
    //    }
    //    os  << endl;
    //}
    //if (hasUp)
    //{
    //    os  << "upper contents:" << endl;
    //    forAll(ldum.upper(), i)
    //    {
    //        os  << "i:" << i << "\t" << ldum.upper()[i] << endl;
    //    }
    //    os  << endl;
    //}

    os.check(FUNCTION_NAME);

    return os;
}


// ************************************************************************* //
