/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::processorLduInterface

Description
    An abstract base class for processor coupled interfaces.

SourceFiles
    processorLduInterface.C
    processorLduInterfaceTemplates.C

\*---------------------------------------------------------------------------*/

#ifndef processorgpuLduInterface_H
#define processorgpuLduInterface_H

#include "gpulduInterface.H"
#include "primitiveFieldsFwd.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                  Class processorLduInterface Declaration
\*---------------------------------------------------------------------------*/

class processorgpuLduInterface
{
    // Private Data

        //- Send buffer.
        //  Only sized and used when compressed or non-blocking comms used.
		mutable List<char> sendBuf_;
        mutable gpuList<char> gpuSendBuf_;

        //- Receive buffer.
        //  Only sized and used when compressed or non-blocking comms used.
        mutable List<char> receiveBuf_;
        mutable gpuList<char> gpuReceiveBuf_;

    // Private Member Functions

        //- Increase buffer size if required
		static void resizeBuf(List<char>& buf, const label size);
        static void resizeBuf(gpuList<char>& buf, const label size);


public:

    //- Runtime type information
    TypeName("processorgpuLduInterface");


    // Constructors

        //- Construct null
        processorgpuLduInterface() = default;


    //- Destructor
    virtual ~processorgpuLduInterface() = default;


    // Member Functions

        // Access

            //- Return communicator used for parallel communication
            virtual label comm() const = 0;

            //- Return processor number (rank in communicator)
            virtual int myProcNo() const = 0;

            //- Return neighbour processor number (rank in communicator)
            virtual int neighbProcNo() const = 0;

            //- Return face transformation tensor
            virtual const tensorgpuField& gpuForwardT() const = 0;

            //- Return message tag used for sending
            virtual int tag() const = 0;


        // Transfer Functions

            //- Raw send function

            template<class Type>
            void send
            (
                const Pstream::commsTypes commsType,
                const gpuList<Type>& f
            ) const;

            //- Raw receive function
            template<class Type>
            void receive
            (
                const Pstream::commsTypes commsType,
                gpuList<Type>& f
            ) const;            

            //- Raw receive function returning field

            template<class Type>
            tmp<gpuField<Type>> receive
            (
                const Pstream::commsTypes commsType,
                const label size
            ) const;

            //- Raw send function with data compression

            template<class Type>
            void compressedSend
            (
                const Pstream::commsTypes commsType,
                const gpuList<Type>& f
            ) const;

            //- Raw receive function with data compression

            template<class Type>
            void compressedReceive
            (
                const Pstream::commsTypes commsType,
                gpuList<Type>& f
            ) const;            

            //- Raw receive function with data compression returning field
            template<class Type>
            tmp<gpuField<Type>> compressedReceive
            (
                const Pstream::commsTypes commsType,
                const label size
            ) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "processorgpuLduInterfaceTemplates.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
