/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2020 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::distributionModels::RosinRammler

Description
    Particle-size distribution model wherein random samples are drawn
    from the doubly-truncated two-parameter Rosin-Rammler (Weibull)
    probability density function:

    \f[
        f(x; \lambda, n, A, B) =
            \frac{
                \frac{n}{\lambda}
                \left( \frac{x}{\lambda} \right)^{n-1}
                \exp\{ -(\frac{x}{\lambda})^n \}
            }{
                \exp\{- (\frac{A}{\lambda})^n \}
              - \exp\{- (\frac{B}{\lambda})^n \}
            }
    \f]
    where

    \vartable
        f(x; \lambda, n, A, B) | Rosin-Rammler probability density function
        \lambda   | Scale parameter
        n         | Shape parameter
        x         | Sample
        A         | Minimum of the distribution
        B         | Maximum of the distribution
    \endvartable

    Constraints:
    - \f$ \infty > B > A > 0\f$
    - \f$ x \in [B,A] \f$
    - \f$ \lambda > 0 \f$
    - \f$ n > 0 \f$

    Random samples are generated by the inverse transform sampling technique
    by using the quantile function of the doubly-truncated two-parameter
    Rosin-Rammler (Weibull) probability density function:

    \f[
        x = \lambda \left( q_{min} - \ln (1 - u r) \right)^{1/n}
    \f]
    with

    \f[
        r = 1 - \exp(-q_{max} + q_{min})
    \f]

    \f[
        q_{min} = \left(\frac{A}{\lambda}\right)^n
    \f]

    \f[
        q_{max} = \left(\frac{B}{\lambda}\right)^n
    \f]
    where \f$ u \f$ is sample drawn from the uniform probability
    density function on the unit interval \f$ (0, 1) \f$.

    Reference:
    \verbatim
    Doubly-truncated two-parameter Weibull distribution and its moments (tag:C):
        Crénin, F. (2015).
        Truncated Weibull Distribution Functions and Moments.
        SSRN 2690255.
        DOI:10.2139/ssrn.2690255
    \endverbatim

Usage
    Minimal example by using \c constant/\<CloudProperties\>:
    \verbatim
    subModels
    {
        injectionModels
        {
            <name>
            {
                ...

                sizeDistribution
                {
                    type        RosinRammler;
                    RosinRammlerDistribution
                    {
                        lambda      <scaleParameterValue>;
                        n           <shapeParameterValue>;
                        minValue    <minValue>;
                        maxValue    <maxValue>;
                    }
                }
            }
        }
    }
    \endverbatim

    where the entries mean:
    \table
      Property     | Description                       | Type | Reqd | Deflt
      type         | Type name: RosinRammler           | word | yes  | -
      RosinRammlerDistribution | Distribution settings | dict | yes  | -
      lambda       | Scale parameter                   | scalar | yes  | -
      n            | Shape parameter                   | scalar | yes  | -
      minValue     | Minimum of the distribution       | scalar | yes  | -
      maxValue     | Maximum of the distribution       | scalar | yes  | -
    \endtable

SourceFiles
    RosinRammler.C

\*---------------------------------------------------------------------------*/

#ifndef RosinRammler_H
#define RosinRammler_H

#include "distributionModel.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace distributionModels
{

/*---------------------------------------------------------------------------*\
                       Class RosinRammler Declaration
\*---------------------------------------------------------------------------*/

class RosinRammler
:
    public distributionModel
{
    // Private Data

        //- Scale parameter
        scalar lambda_;

        //- Shape parameter
        scalar n_;


public:

    //- Runtime type information
    TypeName("RosinRammler");


    // Constructors

        //- Construct from components
        RosinRammler(const dictionary& dict, Random& rndGen);

        //- Copy construct
        RosinRammler(const RosinRammler& p);

        //- Construct and return a clone
        virtual autoPtr<distributionModel> clone() const
        {
            return autoPtr<distributionModel>(new RosinRammler(*this));
        }

        //- No copy assignment
        void operator=(const RosinRammler&) = delete;


    //- Destructor
    virtual ~RosinRammler() = default;


    // Member Functions

        //- Sample the distribution
        virtual scalar sample() const;

        //- Return the theoretical mean of the distribution
        virtual scalar meanValue() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace distributionModels
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
