/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::surfaceInterpolationScheme

Description
    Abstract base class for surface interpolation schemes.

SourceFiles
    surfaceInterpolationScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpusurfaceInterpolationScheme_H
#define gpusurfaceInterpolationScheme_H

#include "tmp.H"
#include "volgpuFieldsFwd.H"
#include "surfacegpuFieldsFwd.H"
#include "typeInfo.H"
#include "runTimeSelectionTables.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

class gpufvMesh;

/*---------------------------------------------------------------------------*\
                 Class surfaceInterpolationScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpusurfaceInterpolationScheme
:
    public refCount
{
    // Private data

        //- Hold reference to mesh
        const gpufvMesh& mesh_;


    // Private Member Functions

        //- No copy construct
        gpusurfaceInterpolationScheme(const gpusurfaceInterpolationScheme&) = delete;

        //- No copy assignment
        void operator=(const gpusurfaceInterpolationScheme&) = delete;


public:

    //- Runtime type information
    TypeName("gpusurfaceInterpolationScheme");


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            gpusurfaceInterpolationScheme,
            Mesh,
            (
                const gpufvMesh& mesh,
                Istream& schemeData
            ),
            (mesh, schemeData)
        );

        declareRunTimeSelectionTable
        (
            tmp,
            gpusurfaceInterpolationScheme,
            MeshFlux,
            (
                const gpufvMesh& mesh,
                const surfaceScalargpuField& faceFlux,
                Istream& schemeData
            ),
            (mesh, faceFlux, schemeData)
        );


    // Constructors

        //- Construct from mesh
        gpusurfaceInterpolationScheme(const gpufvMesh& mesh)
        :
            mesh_(mesh)
        {}


    // Selectors

        //- Return new tmp interpolation scheme
        static tmp<gpusurfaceInterpolationScheme<Type>> New
        (
            const gpufvMesh& mesh,
            Istream& schemeData
        );

        //- Return new tmp interpolation scheme
        static tmp<gpusurfaceInterpolationScheme<Type>> New
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux,
            Istream& schemeData
        );


    //- Destructor
    virtual ~gpusurfaceInterpolationScheme() = default;


    // Member Functions

        //- Return mesh reference
        const gpufvMesh& mesh() const
        {
            return mesh_;
        }


        //- Return the face-interpolate of the given cell field
        //  with the given owner and neighbour weighting factors
        static tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        interpolate
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&,
            const tmp<surfaceScalargpuField>&,
            const tmp<surfaceScalargpuField>&
        );

        //- Return the face-interpolate of the given cell field
        //  with the given weighting factors dotted with given field Sf
        template<class SFType>
        static tmp
        <
            GeometricgpuField
            <
                typename innerProduct<typename SFType::value_type, Type>::type,
                fvsPatchgpuField,
                gpusurfaceMesh
            >
        >
        dotInterpolate
        (
            const SFType& Sf,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf,
            const tmp<surfaceScalargpuField>& tlambdas
        );

        //- Return the face-interpolate of the given cell field
        //  with the given weighting factors
        static tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        interpolate
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&,
            const tmp<surfaceScalargpuField>&
        );

        //- Return the interpolation weighting factors for the given field
        virtual tmp<surfaceScalargpuField> weights
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const = 0;

        //- Return true if this scheme uses an explicit correction
        virtual bool corrected() const
        {
            return false;
        }

        //- Return the explicit correction to the face-interpolate
        //  for the given field
        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        correction(const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&) const
        {
            return tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
            (
                nullptr
            );
        }

        static tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        dotInterpolate
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&,
            const tmp<surfaceScalargpuField>&
        );

        //- Return the face-interpolate of the given cell field
        //  with explicit correction dotted with given field Sf
        virtual
        tmp
        <
            GeometricgpuField
            <
                typename innerProduct<vector, Type>::type,
                fvsPatchgpuField,
                gpusurfaceMesh
            >
        >
        dotInterpolate
        (
            const surfaceVectorgpuField& Sf,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& vf
        ) const;

        //- Return the face-interpolate of the given tmp cell field
        //  with explicit correction dotted with given field Sf
        tmp
        <
            GeometricgpuField
            <
                typename innerProduct<vector, Type>::type,
                fvsPatchgpuField,
                gpusurfaceMesh
            >
        >
        dotInterpolate
        (
            const surfaceVectorgpuField& Sf,
            const tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>&
        ) const;

        //- Return the face-interpolate of the given cell field
        //  with explicit correction
        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        interpolate(const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&) const;

        //- Return the face-interpolate of the given tmp cell field
        //  with explicit correction
        tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        interpolate
        (
            const tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>>&
        ) const;
};


template<>
tmp
<
    GeometricgpuField
    <
        typename innerProduct<vector, scalar>::type,
        fvsPatchgpuField,
        gpusurfaceMesh
    >
>
gpusurfaceInterpolationScheme<scalar>::dotInterpolate
(
    const surfaceVectorgpuField& Sf,
    const GeometricgpuField<scalar, fvPatchgpuField, gpuvolMesh>&
) const;


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makegpuSurfaceInterpolationTypeScheme(SS, Type)                           \
                                                                               \
defineNamedTemplateTypeNameAndDebug(SS<Type>, 0);                              \
                                                                               \
gpusurfaceInterpolationScheme<Type>::addMeshConstructorToTable<SS<Type>>          \
    add##SS##Type##MeshConstructorToTable_;                                    \
                                                                               \
gpusurfaceInterpolationScheme<Type>::addMeshFluxConstructorToTable<SS<Type>>      \
    add##SS##Type##MeshFluxConstructorToTable_;

#define makegpuSurfaceInterpolationScheme(SS)                                     \
                                                                               \
makegpuSurfaceInterpolationTypeScheme(SS, scalar)                                 \
makegpuSurfaceInterpolationTypeScheme(SS, vector)                                 \
makegpuSurfaceInterpolationTypeScheme(SS, sphericalTensor)                        \
makegpuSurfaceInterpolationTypeScheme(SS, symmTensor)                             \
makegpuSurfaceInterpolationTypeScheme(SS, tensor)


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpusurfaceInterpolationScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
