/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011 OpenFOAM Foundation
    Copyright (C) 2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::surfaceInterpolation

Description
    Cell to surface interpolation scheme. Included in fvMesh.

SourceFiles
    surfaceInterpolation.C

\*---------------------------------------------------------------------------*/

#ifndef gpusurfaceInterpolation_H
#define gpusurfaceInterpolation_H

#include "tmp.H"
#include "scalar.H"
#include "volgpuFieldsFwd.H"
#include "surfacegpuFieldsFwd.H"
#include "className.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

class gpufvMesh;
class gpufvGeometryScheme;

/*---------------------------------------------------------------------------*\
                    Class surfaceInterpolation Declaration
\*---------------------------------------------------------------------------*/

class gpusurfaceInterpolation
{
    // Private data

        // Reference to fvMesh
        const gpufvMesh& mesh_;


        // Demand-driven data

            //- Geometry calculation
            mutable tmp<gpufvGeometryScheme> geometryPtr_;

            //- Linear difference weighting factors
            mutable autoPtr<surfaceScalargpuField> weights_;

            //- Cell-centre difference coefficients
            mutable autoPtr<surfaceScalargpuField> deltaCoeffs_;

            //- Non-orthogonal cell-centre difference coefficients
            mutable autoPtr<surfaceScalargpuField> nonOrthDeltaCoeffs_;

            //- Non-orthogonality correction vectors
            mutable autoPtr<surfaceVectorgpuField> nonOrthCorrectionVectors_;


protected:

    // Protected Member Functions

        // Storage management

            //- Clear all geometry and addressing
            void clearOut();


public:

    // Declare name of the class and its debug switch
    ClassName("gpusurfaceInterpolation");


    // Constructors

        //- Construct given an fvMesh
        explicit gpusurfaceInterpolation(const gpufvMesh&);


    //- Destructor
    virtual ~gpusurfaceInterpolation();


    // Member functions

        //- Return reference to geometry calculation scheme
        virtual const gpufvGeometryScheme& geometry() const;

        //- Set geometry calculation scheme
        void geometry(tmp<gpufvGeometryScheme>&);

        //- Return reference to linear difference weighting factors
        virtual const surfaceScalargpuField& weights() const;

        //- Return reference to cell-centre difference coefficients
        virtual const surfaceScalargpuField& deltaCoeffs() const;

        //- Return reference to non-orthogonal cell-centre difference
        //  coefficients
        virtual const surfaceScalargpuField& nonOrthDeltaCoeffs() const;

        //- Return reference to non-orthogonality correction vectors
        virtual const surfaceVectorgpuField& nonOrthCorrectionVectors() const;

        //- Do what is necessary if the mesh has moved
        virtual bool movePoints();

        //- Update all geometric data
        virtual void updateGeom();
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
