/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fvMeshLduAddressing

Description
    Foam::fvMeshLduAddressing

SourceFiles
    fvMeshLduAddressing.C

\*---------------------------------------------------------------------------*/

#ifndef gpufvMeshLduAddressing_H
#define gpufvMeshLduAddressing_H

#include "gpulduAddressing.H"
#include "gpufvMesh.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                        Class fvMeshLduAddressing Declaration
\*---------------------------------------------------------------------------*/

class gpufvMeshLduAddressing
:
    public gpulduAddressing
{
    // Private Data

        //- Lower as a subList of allOwner
        const labelgpuList lowerAddr_;

        //- Upper as a reference to neighbour
        const labelgpuList& upperAddr_;

        //- Patch addressing as a list of sublists
        const UPtrList<const labelgpuList> patchAddr_;

        //- Patch field evaluation schedule
        const lduSchedule& patchSchedule_;


    // Private Member Functions

        //- No copy construct
        gpufvMeshLduAddressing(const gpufvMeshLduAddressing&) = delete;

        //- No copy assignment
        void operator=(const gpufvMeshLduAddressing&) = delete;


public:

    // Constructors

        //- Construct from components
        explicit gpufvMeshLduAddressing(const gpufvMesh& mesh)
        :
            gpulduAddressing(mesh.hostmesh().nCells()),
            lowerAddr_
            (
                mesh.hostmesh().gpuFaceOwner(),
                mesh.hostmesh().nInternalFaces()
            ),
            upperAddr_(mesh.hostmesh().gpuFaceNeighbour()),
            patchAddr_(mesh.boundary().gpuFaceCells()),
            patchSchedule_(mesh.hostmesh().globalData().patchSchedule())
        {}


    //- Destructor
    ~gpufvMeshLduAddressing() = default;


    // Member Functions

        //- Return lower addressing (i.e. lower label = upper triangle)
        const labelgpuList& lowerAddr() const noexcept
        {
            return lowerAddr_;
        }

        //- Return upper addressing (i.e. upper label)
        const labelgpuList& upperAddr() const noexcept
        {
            return upperAddr_;
        }

        //- Return patch addressing
        const labelgpuList& patchAddr(const label patchi) const
        {
            return patchAddr_[patchi];
        }

        //- Return patch field evaluation schedule
        const lduSchedule& patchSchedule() const noexcept
        {
            return patchSchedule_;
        }

//********************************************************************//

        //- Return wheather patch addressing is available
        bool patchAvailable(const label i) const
        {
            return true;
        }

        label nPatches() const
        {
            return patchAddr_.size();
        }	
	
        label level() const
        {
            return 0;
        }

};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
