/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2015-2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "IOobject.H"
#include "dictionary.H"
#include "gpufvMesh.H"
#include "fvPatchgpuFieldMapper.H"
#include "gpuvolMesh.H"

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class Type>
Foam::fvPatchgpuField<Type>::fvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    gpuField<Type>(p.size()),
    patch_(p),
    internalField_(iF),
    updated_(false),
    manipulatedMatrix_(false),
    useImplicit_(false),
    patchType_()
{}


template<class Type>
Foam::fvPatchgpuField<Type>::fvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const Type& value
)
:
    gpuField<Type>(p.size(), value),
    patch_(p),
    internalField_(iF),
    updated_(false),
    manipulatedMatrix_(false),
    useImplicit_(false),
    patchType_()
{}


template<class Type>
Foam::fvPatchgpuField<Type>::fvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const word& patchType
)
:
    gpuField<Type>(p.size()),
    patch_(p),
    internalField_(iF),
    updated_(false),
    manipulatedMatrix_(false),
    useImplicit_(false),
    patchType_(patchType)
{}


template<class Type>
Foam::fvPatchgpuField<Type>::fvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const Field<Type>& f
)
:
    gpuField<Type>(f),
    patch_(p),
    internalField_(iF),
    updated_(false),
    manipulatedMatrix_(false),
    useImplicit_(false),
    patchType_()
{}

template<class Type>
Foam::fvPatchgpuField<Type>::fvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const gpuField<Type>& f
)
:
    gpuField<Type>(f),
    patch_(p),
    internalField_(iF),
    updated_(false),
    manipulatedMatrix_(false),
    useImplicit_(false),
    patchType_()
{}

template<class Type>
Foam::fvPatchgpuField<Type>::fvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const dictionary& dict,
    const bool valueRequired
)
:
    gpuField<Type>(p.size()),
    patch_(p),
    internalField_(iF),
    updated_(false),
    manipulatedMatrix_(false),
    useImplicit_(dict.getOrDefault<bool>("useImplicit", false)),
    patchType_(dict.getOrDefault<word>("patchType", word::null))
{
    if (valueRequired)
    {
        if (dict.found("value"))
        {
            gpuField<Type>::operator=
            (
                gpuField<Type>("value", dict, p.size())
            );
        }
        else
        {
            FatalIOErrorInFunction(dict)
                << "Essential entry 'value' missing on patch " << p.name() << nl
                << exit(FatalIOError);
        }
    }
}


template<class Type>
Foam::fvPatchgpuField<Type>::fvPatchgpuField
(
    const fvPatchgpuField<Type>& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    gpuField<Type>(p.size()),
    patch_(p),
    internalField_(iF),
    updated_(false),
    manipulatedMatrix_(false),
    useImplicit_(ptf.useImplicit_),
    patchType_(ptf.patchType_)
{
    // For unmapped faces set to internal field value (zero-gradient)
    if (notNull(iF) && mapper.hasUnmapped())
    {
        fvPatchgpuField<Type>::operator=(this->patchInternalField());
    }
    this->map(ptf, mapper);
}


template<class Type>
Foam::fvPatchgpuField<Type>::fvPatchgpuField
(
    const fvPatchgpuField<Type>& ptf
)
:
    gpuField<Type>(ptf),
    patch_(ptf.patch_),
    internalField_(ptf.internalField_),
    updated_(false),
    manipulatedMatrix_(false),
    useImplicit_(ptf.useImplicit_),
    patchType_(ptf.patchType_)
{}


template<class Type>
Foam::fvPatchgpuField<Type>::fvPatchgpuField
(
    const fvPatchgpuField<Type>& ptf,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    gpuField<Type>(ptf),
    patch_(ptf.patch_),
    internalField_(iF),
    updated_(false),
    manipulatedMatrix_(false),
    useImplicit_(ptf.useImplicit_),
    patchType_(ptf.patchType_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
const Foam::objectRegistry& Foam::fvPatchgpuField<Type>::db() const
{
    return patch_.boundaryMesh().mesh().hostmesh();
}


template<class Type>
void Foam::fvPatchgpuField<Type>::check(const fvPatchgpuField<Type>& ptf) const
{
    if (&patch_ != &(ptf.patch_))
    {
        FatalErrorInFunction
            << "different patches for fvPatchgpuField<Type>s"
            << abort(FatalError);
    }
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>> Foam::fvPatchgpuField<Type>::snGrad() const
{
    return patch_.deltaCoeffs()*(*this - patchInternalField());
}


template<class Type>
Foam::tmp<Foam::gpuField<Type>>
Foam::fvPatchgpuField<Type>::patchInternalField() const
{
    return patch_.patchInternalField(internalField_.field());
}


template<class Type>
void Foam::fvPatchgpuField<Type>::patchInternalField(gpuField<Type>& pif) const
{
    patch_.patchInternalField(internalField_.field(), pif);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::autoMap
(
    const fvPatchgpuFieldMapper& mapper
)
{
/*
    gpuField<Type>& f = *this;

    if (!this->size() && !mapper.distributed())
    {
        f.setSize(mapper.size());
        if (f.size())
        {
            f = this->patchInternalField();
        }
    }
    else
    {
        // Map all faces provided with mapping data
        gpuField<Type>::autoMap(mapper);


        // For unmapped faces set to internal field value (zero-gradient)
        if (mapper.hasUnmapped())
        {
            gpuField<Type> pif(this->patchInternalField());

            if
            (
                mapper.direct()
             && notNull(mapper.directAddressing())
             && mapper.directAddressing().size()
            )
            {
                const labelList& mapAddressing = mapper.directAddressing();

                forAll(mapAddressing, i)
                {
                    if (mapAddressing[i] < 0)
                    {
                        f[i] = pif[i];
                    }
                }
            }
            else if (!mapper.direct() && mapper.addressing().size())
            {
                const labelListList& mapAddressing = mapper.addressing();

                forAll(mapAddressing, i)
                {
                    const labelList& localAddrs = mapAddressing[i];

                    if (!localAddrs.size())
                    {
                        f[i] = pif[i];
                    }
                }
            }
        }
    }
*/
}


template<class Type>
void Foam::fvPatchgpuField<Type>::rmap
(
    const fvPatchgpuField<Type>& ptf,
    const labelgpuList& addr
)
{
    gpuField<Type>::rmap(ptf, addr);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::updateCoeffs()
{
    updated_ = true;
}


template<class Type>
void Foam::fvPatchgpuField<Type>::updateWeightedCoeffs(const scalargpuField& weights)
{
    // Default behaviour ignores the weights
    if (!updated_)
    {
        updateCoeffs();

        updated_ = true;
    }
}


template<class Type>
void Foam::fvPatchgpuField<Type>::evaluate(const Pstream::commsTypes)
{
    if (!updated_)
    {
        updateCoeffs();
    }

    updated_ = false;
    manipulatedMatrix_ = false;
}


template<class Type>
void Foam::fvPatchgpuField<Type>::manipulateMatrix(gpufvMatrix<Type>& matrix)
{
    manipulatedMatrix_ = true;
}


template<class Type>
void Foam::fvPatchgpuField<Type>::manipulateMatrix
(
    gpufvMatrix<Type>& matrix,
    const scalargpuField& weights
)
{
    manipulatedMatrix_ = true;
}


template<class Type>
void Foam::fvPatchgpuField<Type>::manipulateMatrix
(
    gpufvMatrix<Type>& matrix,
    const label iMatrix,
    const direction cmp
)
{
    manipulatedMatrix_ = true;
}


template<class Type>
void Foam::fvPatchgpuField<Type>::write(Ostream& os) const
{
    os.writeEntry("type", type());

    if (useImplicit_)
    {
        os.writeEntry("useImplicit", "true");
    }

    if (patchType_.size())
    {
        os.writeEntry("patchType", patchType_);
    }
}


// * * * * * * * * * * * * * * * Member Operators  * * * * * * * * * * * * * //

template<class Type>
void Foam::fvPatchgpuField<Type>::operator=
(
    const UList<Type>& ul
)
{
    gpuField<Type>::operator=(ul);
}

template<class Type>
void Foam::fvPatchgpuField<Type>::operator=
(
    const gpuList<Type>& ul
)
{
    gpuField<Type>::operator=(ul);
}

template<class Type>
void Foam::fvPatchgpuField<Type>::operator=
(
    const fvPatchgpuField<Type>& ptf
)
{
    check(ptf);
    gpuField<Type>::operator=(ptf);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator+=
(
    const fvPatchgpuField<Type>& ptf
)
{
    check(ptf);
    gpuField<Type>::operator+=(ptf);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator-=
(
    const fvPatchgpuField<Type>& ptf
)
{
    check(ptf);
    gpuField<Type>::operator-=(ptf);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator*=
(
    const fvPatchgpuField<scalar>& ptf
)
{
    if (&patch_ != &ptf.patch())
    {
        FatalErrorInFunction
            << "incompatible patches for patch fields"
            << abort(FatalError);
    }

    gpuField<Type>::operator*=(ptf);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator/=
(
    const fvPatchgpuField<scalar>& ptf
)
{
    if (&patch_ != &ptf.patch())
    {
        FatalErrorInFunction
            << abort(FatalError);
    }

    gpuField<Type>::operator/=(ptf);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator+=
(
    const gpuField<Type>& tf
)
{
    gpuField<Type>::operator+=(tf);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator-=
(
    const gpuField<Type>& tf
)
{
    gpuField<Type>::operator-=(tf);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator*=
(
    const scalargpuField& tf
)
{
    gpuField<Type>::operator*=(tf);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator/=
(
    const scalargpuField& tf
)
{
    gpuField<Type>::operator/=(tf);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator=
(
    const Type& t
)
{
    gpuField<Type>::operator=(t);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator+=
(
    const Type& t
)
{
    gpuField<Type>::operator+=(t);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator-=
(
    const Type& t
)
{
    gpuField<Type>::operator-=(t);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator*=
(
    const scalar s
)
{
    gpuField<Type>::operator*=(s);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator/=
(
    const scalar s
)
{
    gpuField<Type>::operator/=(s);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator==
(
    const fvPatchgpuField<Type>& ptf
)
{
    gpuField<Type>::operator=(ptf);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator==
(
    const gpuField<Type>& tf
)
{
    gpuField<Type>::operator=(tf);
}


template<class Type>
void Foam::fvPatchgpuField<Type>::operator==
(
    const Type& t
)
{
    gpuField<Type>::operator=(t);
}


// * * * * * * * * * * * * * * * IOstream Operators  * * * * * * * * * * * * //

template<class Type>
Foam::Ostream& Foam::operator<<(Ostream& os, const fvPatchgpuField<Type>& ptf)
{
    ptf.write(os);

    os.check(FUNCTION_NAME);

    return os;
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#include "fvPatchgpuFieldNew.C"

// ************************************************************************* //
