/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2017-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "inletOutletFvPatchgpuField.H"

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class Type>
Foam::inletOutletFvPatchgpuField<Type>::inletOutletFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    mixedFvPatchgpuField<Type>(p, iF),
    phiName_("phi")
{
    this->refValue() = Zero;
    this->refGrad() = Zero;
    this->valueFraction() = 0.0;
}


template<class Type>
Foam::inletOutletFvPatchgpuField<Type>::inletOutletFvPatchgpuField
(
    const inletOutletFvPatchgpuField<Type>& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    mixedFvPatchgpuField<Type>(ptf, p, iF, mapper),
    phiName_(ptf.phiName_)
{}


template<class Type>
Foam::inletOutletFvPatchgpuField<Type>::inletOutletFvPatchgpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<Type, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    mixedFvPatchgpuField<Type>(p, iF),
    phiName_(dict.getOrDefault<word>("phi", "phi"))
{
    this->patchType() = dict.getOrDefault<word>("patchType", word::null);

    this->refValue() = gpuField<Type>("inletValue", dict, p.size());

    if (dict.found("value"))
    {
        fvPatchgpuField<Type>::operator=
        (
            gpuField<Type>("value", dict, p.size())
        );
    }
    else
    {
        fvPatchgpuField<Type>::operator=(this->refValue());
    }

    this->refGrad() = Zero;
    this->valueFraction() = 0.0;
}


template<class Type>
Foam::inletOutletFvPatchgpuField<Type>::inletOutletFvPatchgpuField
(
    const inletOutletFvPatchgpuField<Type>& ptf
)
:
    mixedFvPatchgpuField<Type>(ptf),
    phiName_(ptf.phiName_)
{}


template<class Type>
Foam::inletOutletFvPatchgpuField<Type>::inletOutletFvPatchgpuField
(
    const inletOutletFvPatchgpuField<Type>& ptf,
    const DimensionedgpuField<Type, gpuvolMesh>& iF
)
:
    mixedFvPatchgpuField<Type>(ptf, iF),
    phiName_(ptf.phiName_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
void Foam::inletOutletFvPatchgpuField<Type>::updateCoeffs()
{
    if (this->updated())
    {
        return;
    }

    const gpuField<scalar>& phip =
        this->patch().template lookupPatchField<surfaceScalargpuField, scalar>
        (
            phiName_
        );

    this->valueFraction() = 1.0 - pos0(phip);

    mixedFvPatchgpuField<Type>::updateCoeffs();
}


template<class Type>
void Foam::inletOutletFvPatchgpuField<Type>::write(Ostream& os) const
{
    fvPatchgpuField<Type>::write(os);
    os.writeEntryIfDifferent<word>("phi", "phi", phiName_);
    this->refValue().writeEntry("inletValue", os);
    this->writeEntry("value", os);
}


// * * * * * * * * * * * * * * * Member Operators  * * * * * * * * * * * * * //

template<class Type>
void Foam::inletOutletFvPatchgpuField<Type>::operator=
(
    const fvPatchgpuField<Type>& ptf
)
{
    fvPatchgpuField<Type>::operator=
    (
        this->valueFraction()*this->refValue()
        + (1 - this->valueFraction())*ptf
    );
}


// ************************************************************************* //
