/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2021 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::limitedSnGrad

Group
    grpFvSnGradSchemes

Description
    Surface gradient scheme with limited explicit non-orthogonal correction.

    The limiter is controlled by a coefficient with a value between 0 and 1
    which when 0 switches the correction off and the scheme behaves as
    \c uncorrected \c snGrad, when set to 1 the full correction of the
    selected scheme is used and the scheme behaves as \c corrected \c snGrad,
    and when set to 0.5 the limiter is calculated such that the non-orthogonal
    component does not exceed the orthogonal component.

Usage
    Minimal example by using \c system/fvSchemes:
    \verbatim
    snGradSchemes
    {
        snGrad(<term>)       limited <corrected scheme> <coefficient>;

        // Backward compatibility
        snGrad(<term>)       limited <coefficient>;
    }
    \endverbatim

SourceFiles
    limitedSnGrad.C

\*---------------------------------------------------------------------------*/

#ifndef gpulimitedSnGrad_H
#define gpulimitedSnGrad_H

#include "gpucorrectedSnGrad.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                        Class limitedSnGrad Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpulimitedSnGrad
:
    public gpusnGradScheme<Type>
{
    // Private Data

        //- Type of correction scheme
        tmp<gpusnGradScheme<Type>> correctedScheme_;

        //- Limiter coefficient
        scalar limitCoeff_;


    // Private Member Functions

        //- No copy assignment
        void operator=(const gpulimitedSnGrad&) = delete;

        //- Lookup function for the corrected to support backward compatibility
        //- of dictionary specification
        tmp<gpusnGradScheme<Type>> lookupCorrectedScheme(Istream& schemeData)
        {
            token nextToken(schemeData);

            if (nextToken.isNumber())
            {
                limitCoeff_ = nextToken.number();
                return tmp<gpusnGradScheme<Type>>
                (
                    new gpucorrectedSnGrad<Type>(this->mesh())
                );
            }
            else
            {
                schemeData.putBack(nextToken);
                tmp<gpusnGradScheme<Type>> tcorrectedScheme
                (
                    fv::gpusnGradScheme<Type>::New(this->mesh(), schemeData)
                );

                schemeData >> limitCoeff_;

                return tcorrectedScheme;
            }
        }


public:

    //- Runtime type information
    TypeName("limited");


    // Constructors

        //- Construct from mesh
        gpulimitedSnGrad(const gpufvMesh& mesh)
        :
            gpusnGradScheme<Type>(mesh),
            correctedScheme_(new gpucorrectedSnGrad<Type>(this->mesh())),
            limitCoeff_(1)
        {}

        //- Construct from mesh and data stream
        gpulimitedSnGrad(const gpufvMesh& mesh, Istream& schemeData)
        :
            gpusnGradScheme<Type>(mesh),
            correctedScheme_(lookupCorrectedScheme(schemeData))
        {
            if (limitCoeff_ < 0 || limitCoeff_ > 1)
            {
                FatalIOErrorInFunction(schemeData)
                    << "limitCoeff is specified as " << limitCoeff_
                    << " but should be >= 0 && <= 1"
                    << exit(FatalIOError);
            }
        }


    //- Destructor
    virtual ~gpulimitedSnGrad() = default;


    // Member Functions

        //- Return the interpolation weighting factors for the given field
        virtual tmp<surfaceScalargpuField> deltaCoeffs
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const
        {
            return this->mesh().nonOrthDeltaCoeffs();
        }

        //- Return true if this scheme uses an explicit correction
        virtual bool corrected() const noexcept
        {
            return true;
        }

        //- Return the explicit correction to the limitedSnGrad
        //- for the given field
        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        correction(const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpulimitedSnGrad.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
