/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::CoEulerDdtScheme

Group
    grpFvDdtSchemes

Description
    Courant number limited first-order Euler implicit/explicit ddt.

    The time-step is adjusted locally so that the local Courant number
    does not exceed the specified value.

    This scheme should only be used for steady-state computations
    using transient codes where local time-stepping is preferable to
    under-relaxation for transport consistency reasons.

SourceFiles
    CoEulerDdtScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpuCoEulerDdtScheme_H
#define gpuCoEulerDdtScheme_H

#include "gpuddtScheme.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                       Class CoEulerDdtScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpuCoEulerDdtScheme
:
    public fv::gpuddtScheme<Type>
{
    // Private Data

        //- Name of the flux field used to calculate the local time-step
        word phiName_;

        //- Name of the density field used to obtain the volumetric flux
        //  from the mass flux if required
        word rhoName_;

        //- Maximum local Courant number
        scalar maxCo_;


    // Private Member Functions

        //- No copy construct
        gpuCoEulerDdtScheme(const gpuCoEulerDdtScheme&) = delete;

        //- No copy assignment
        void operator=(const gpuCoEulerDdtScheme&) = delete;

        //- Return the reciprocal of the Courant-number limited time-step
        tmp<volScalargpuField> CorDeltaT() const;

        //- Return the reciprocal of the face-Courant-number limited time-step
        tmp<surfaceScalargpuField> CofrDeltaT() const;


public:

    //- Runtime type information
    TypeName("CoEuler");


    // Constructors

        //- Construct from mesh and Istream
        gpuCoEulerDdtScheme(const gpufvMesh& mesh, Istream& is)
        :
            gpuddtScheme<Type>(mesh, is),
            phiName_(is),
            rhoName_(is),
            maxCo_(readScalar(is))
        {}


    // Member Functions

        //- Return mesh reference
        const gpufvMesh& mesh() const
        {
            return fv::gpuddtScheme<Type>::mesh();
        }

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const dimensioned<Type>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const dimensionedScalar&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const volScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDdt
        (
            const volScalargpuField& alpha,
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& psi
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const dimensionedScalar&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const volScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );

        tmp<gpufvMatrix<Type>> fvmDdt
        (
            const volScalargpuField& alpha,
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& psi
        );

        typedef typename gpuddtScheme<Type>::fluxFieldType fluxFieldType;

        tmp<fluxFieldType> fvcDdtUfCorr
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
        );

        tmp<fluxFieldType> fvcDdtPhiCorr
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi
        );

        tmp<fluxFieldType> fvcDdtUfCorr
        (
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>& Uf
        );

        tmp<fluxFieldType> fvcDdtPhiCorr
        (
            const volScalargpuField& rho,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>& U,
            const fluxFieldType& phi
        );

        tmp<surfaceScalargpuField> meshPhi
        (
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        );
};


template<>
tmp<surfaceScalargpuField> gpuCoEulerDdtScheme<scalar>::fvcDdtUfCorr
(
    const GeometricgpuField<scalar, fvPatchgpuField, gpuvolMesh>& U,
    const GeometricgpuField<scalar, fvsPatchgpuField, gpusurfaceMesh>& Uf
);

template<>
tmp<surfaceScalargpuField> gpuCoEulerDdtScheme<scalar>::fvcDdtPhiCorr
(
    const volScalargpuField& U,
    const surfaceScalargpuField& phi
);

template<>
tmp<surfaceScalargpuField> gpuCoEulerDdtScheme<scalar>::fvcDdtUfCorr
(
    const volScalargpuField& rho,
    const volScalargpuField& U,
    const surfaceScalargpuField& Uf
);

template<>
tmp<surfaceScalargpuField> gpuCoEulerDdtScheme<scalar>::fvcDdtPhiCorr
(
    const volScalargpuField& rho,
    const volScalargpuField& U,
    const surfaceScalargpuField& phi
);


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpuCoEulerDdtScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
