/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fv::convectionScheme

Group
    grpFvConvectionSchemes

Description
    Abstract base class for convection schemes.

SourceFiles
    convectionScheme.C

\*---------------------------------------------------------------------------*/

#ifndef gpuconvectionScheme_H
#define gpuconvectionScheme_H

#include "tmp.H"
#include "volgpuFieldsFwd.H"
#include "surfacegpuFieldsFwd.H"
#include "typeInfo.H"
#include "runTimeSelectionTables.H"
#include "gpumultivariateSurfaceInterpolationScheme.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

template<class Type>
class gpufvMatrix;

class gpufvMesh;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace fv
{

/*---------------------------------------------------------------------------*\
                           Class convectionScheme Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class gpuconvectionScheme
:
    public refCount
{
    // Private data

        const gpufvMesh& mesh_;


public:

    //- Runtime type information
    virtual const word& type() const = 0;


    // Declare run-time constructor selection tables

        declareRunTimeSelectionTable
        (
            tmp,
            gpuconvectionScheme,
            Istream,
            (
                const gpufvMesh& mesh,
                const surfaceScalargpuField& faceFlux,
                Istream& schemeData
            ),
            (mesh, faceFlux, schemeData)
        );

        declareRunTimeSelectionTable
        (
            tmp,
            gpuconvectionScheme,
            Multivariate,
            (
                const gpufvMesh& mesh,
                const typename gpumultivariateSurfaceInterpolationScheme<Type>::
                    fieldTable& fields,
                const surfaceScalargpuField& faceFlux,
                Istream& schemeData
            ),
            (mesh, fields, faceFlux, schemeData)
        );


    // Constructors

        //- Copy construct
        gpuconvectionScheme(const gpuconvectionScheme&);

        //- Construct from mesh, flux and Istream
        gpuconvectionScheme
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField&
        )
        :
            mesh_(mesh)
        {}


    // Selectors

        //- Return a pointer to a new convectionScheme created on freestore
        static tmp<gpuconvectionScheme<Type>> New
        (
            const gpufvMesh& mesh,
            const surfaceScalargpuField& faceFlux,
            Istream& schemeData
        );


        //- Return a pointer to a new multivariate convectionScheme
        //  created on freestore
        static tmp<gpuconvectionScheme<Type>> New
        (
            const gpufvMesh& mesh,
            const typename gpumultivariateSurfaceInterpolationScheme<Type>::
                fieldTable& fields,
            const surfaceScalargpuField& faceFlux,
            Istream& schemeData
        );


    //- Destructor
    virtual ~gpuconvectionScheme();


    // Member Functions

        //- Return mesh reference
        const gpufvMesh& mesh() const
        {
            return mesh_;
        }

        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>>
        interpolate
        (
            const surfaceScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const = 0;

        virtual tmp<GeometricgpuField<Type, fvsPatchgpuField, gpusurfaceMesh>> flux
        (
            const surfaceScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const = 0;

        virtual tmp<gpufvMatrix<Type>> fvmDiv
        (
            const surfaceScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const = 0;

        virtual tmp<GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>> fvcDiv
        (
            const surfaceScalargpuField&,
            const GeometricgpuField<Type, fvPatchgpuField, gpuvolMesh>&
        ) const = 0;


    // Member operators

        void operator=(const gpuconvectionScheme<Type>&);
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fv

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Add the patch constructor functions to the hash tables

#define makeFvgpuConvectionTypeScheme(SS, Type)                                   \
    defineNamedTemplateTypeNameAndDebug(Foam::fv::SS<Foam::Type>, 0);          \
                                                                               \
    namespace Foam                                                             \
    {                                                                          \
        namespace fv                                                           \
        {                                                                      \
            gpuconvectionScheme<Type>::addIstreamConstructorToTable<SS<Type>>     \
                add##SS##Type##IstreamConstructorToTable_;                     \
        }                                                                      \
    }

#define makeFvgpuConvectionScheme(SS)                                             \
                                                                               \
makeFvgpuConvectionTypeScheme(SS, scalar)                                         \
makeFvgpuConvectionTypeScheme(SS, vector)                                         \
makeFvgpuConvectionTypeScheme(SS, sphericalTensor)                                \
makeFvgpuConvectionTypeScheme(SS, symmTensor)                                     \
makeFvgpuConvectionTypeScheme(SS, tensor)


#define makeMultivariateFvgpuConvectionTypeScheme(SS, Type)                       \
    defineNamedTemplateTypeNameAndDebug(Foam::fv::SS<Foam::Type>, 0);          \
                                                                               \
    namespace Foam                                                             \
    {                                                                          \
        namespace fv                                                           \
        {                                                                      \
            gpuconvectionScheme<Type>::                                           \
                addMultivariateConstructorToTable<SS<Type>>                    \
                add##SS##Type##MultivariateConstructorToTable_;                \
        }                                                                      \
    }


#define makeMultivariateFvgpuConvectionScheme(SS)                                 \
                                                                               \
makeMultivariateFvgpuConvectionTypeScheme(SS, scalar)                             \
makeMultivariateFvgpuConvectionTypeScheme(SS, vector)                             \
makeMultivariateFvgpuConvectionTypeScheme(SS, sphericalTensor)                    \
makeMultivariateFvgpuConvectionTypeScheme(SS, symmTensor)                         \
makeMultivariateFvgpuConvectionTypeScheme(SS, tensor)


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "gpuconvectionScheme.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
