/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2016-2019 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::lduAddressing

Description
    The class contains the addressing required by the lduMatrix: upper, lower
    and losort.

    The addressing can be created in two ways: either with references to
    upper and lower in which case it stores references or from labelLists,
    in which case it stores the addressing itself. Additionally, the losort
    addressing belongs to the class is as on lazy evaluation.

    The ordering of owner addresses is such that the labels are in
    increasing order, with groups of identical labels for edges "owned" by
    the same point. The neighbour labels are also ordered in ascending
    order but only for groups of edges belonging to each point. An example
    is given below:
    \verbatim
        owner     neighbour
        0         1
        0         20
        1         2
        1         21
        2         3
        2         22
        3         4
        3         23
        4         5
        4         24
        5         6
        5         25
        6         7
        6         26
        7         8
        7         27
        8         9
        8         28
        9         10
        9         29
    \endverbatim

    There exists an alternative way of addressing the owner
    list: instead of repeating the same label in the owner list, it is
    possible to address the start of each point neighbours in the
    neighbour list. This reduces the size of owner addressing from a list
    over all edges to a list over all points + 1:

    \verbatim
        Owner start list: 0 2 4 6 8 10 12 14 16 18
    \endverbatim

    We shall use the second form of the addressing for fast lookup
    of edge label from the known owner and neighbour, using the following
    algorithm:
    -# take the owner label and position the start of lookup
       using the owner start list
    -# loop through all neighbours of this owner (ending at the start of
      lookup of owner + 1) until the match with current neighbour is found.
      The index used on the neighbour list for the match is the edge index.

    While owner start addressing allows us to find the edge owned by the
    points, it is also necessary to find the edges for which the point is
    a neighbour. Losort addressing lists the edges neighboured by the
    point and we shall use the same trick as above to address into this
    list. Thus, for every point the losort start gives the address of the
    first face to neighbour this point.

SourceFiles
    lduAddressing.C

\*---------------------------------------------------------------------------*/

#ifndef gpulduAddressing_H
#define gpulduAddressing_H

#include "labelList.H"
#include "lduSchedule.H"
#include "Tuple2.H"

#include <thrust/iterator/permutation_iterator.h>
#include <thrust/iterator/counting_iterator.h>
#include <thrust/iterator/constant_iterator.h>
#include <thrust/iterator/transform_iterator.h>
#include <thrust/sort.h>

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class lduAddressing Declaration
\*---------------------------------------------------------------------------*/

class gpulduAddressing
{
    // Private Data

        //- Number of equations
        label size_;


    //- Demand-driven data

        //- Losort addressing
        mutable labelgpuList* losortAddr_;

        //- Owner start addressing
        mutable labelgpuList* ownerStartAddr_;

        //- Losort start addressing
        mutable labelgpuList* losortStartAddr_;
	
        //- Owner sort addressing
        mutable labelgpuList* gOwnerSortAddr_;

        mutable PtrList<const labelgpuList> gPatchSortCells_;

        mutable PtrList<const labelgpuList> gPatchSortAddr_;

        mutable PtrList<const labelgpuList> gPatchSortStartAddr_;

    // Private Member Functions

        //- No copy construct
        gpulduAddressing(const gpulduAddressing&) = delete;

        //- No copy assignment
        void operator=(const gpulduAddressing&) = delete;

        //- Calculate losort
        void calcLosort() const;

        //- Calculate owner start
        void calcOwnerStart() const;

        //- Calculate losort start
        void calcLosortStart() const;

        //- Calculate patch sort
        void calcPatchSort() const;
		
        //- Calculate patch sort start
        void calcPatchSortStart() const;

public:

    //- Construct with size (number of equations)
    explicit gpulduAddressing(const label nEqns)
    :
        size_(nEqns),
        losortAddr_(nullptr),
        losortStartAddr_(nullptr),
        ownerStartAddr_(nullptr),
        gOwnerSortAddr_(nullptr)
    {}


    //- Destructor
    virtual ~gpulduAddressing();


    // Member Functions

        //- Return number of equations
        label size() const
        {
            return size_;
        }

        virtual label level() const = 0;

        virtual label nPatches() const = 0;
        virtual bool patchAvailable
        (
            const label patchNo
        ) const = 0;

        //- Return lower addressing
        virtual const labelgpuList& lowerAddr() const = 0;

        //- Return upper addressing
        virtual const labelgpuList& upperAddr() const = 0;

        //- Return patch to internal addressing given patch number

        virtual const labelgpuList& patchAddr
        (
            const label patchNo
        ) const = 0;

        const labelgpuList& gpuPatchSortCells
        (
            const label patchNo
        ) const;

        const labelgpuList& gpuPatchSortAddr
        (
            const label patchNo
        ) const;

        const labelgpuList& gpuPatchSortStartAddr
        (
            const label patchNo
        ) const;

        //- Return patch field evaluation schedule
        virtual const lduSchedule& patchSchedule() const = 0;

        //- Clear additional addressing
        void clearOut();

        //- Return losort addressing
        const labelgpuList& losortAddr() const;

        //- Return owner start addressing
        const labelgpuList& ownerStartAddr() const;

        //- Return losort start addressing
        const labelgpuList& losortStartAddr() const;

        //- Return owner sort addressing
        const labelgpuList& gpuOwnerSortAddr() const;

        //- Return off-diagonal index given owner and neighbour label
        label triIndex(const label a, const label b) const;

        //- Calculate bandwidth and profile of addressing
        Tuple2<label, scalar> band() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
