/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2013-2015 OpenFOAM Foundation
    Copyright (C) 2019-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::turbulenceModel

Group
    grpTurbulence

Description
    Abstract base class for turbulence models (RAS, LES and laminar).

SourceFiles
    turbulenceModel.C

\*---------------------------------------------------------------------------*/

#ifndef turbulenceModel_H
#define turbulenceModel_H

#include "IOdictionary.H"
#include "primitiveFieldsFwd.H"
#include "volgpuFieldsFwd.H"
#include "surfacegpuFieldsFwd.H"
#include "gpufvMatricesFwd.H"
#include "gpunearWallDist.H"
#include "geometricOneField.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward declarations
class gpufvMesh;

/*---------------------------------------------------------------------------*\
                           Class turbulenceModel Declaration
\*---------------------------------------------------------------------------*/

class turbulenceModel
:
    public IOdictionary
{

protected:

    // Protected data

        const Time& runTime_;
        const gpufvMesh& mesh_;

        const volVectorgpuField& U_;
        const surfaceScalargpuField& alphaRhoPhi_;
        const surfaceScalargpuField& phi_;

        //- Near wall distance boundary field
        gpunearWallDist y_;


private:

    // Private Member Functions

        //- No copy construct
        turbulenceModel(const turbulenceModel&) = delete;

        //- No copy assignment
        void operator=(const turbulenceModel&) = delete;


public:

    //- Runtime type information
    TypeName("turbulenceModel");

    //- Default name of the turbulence properties dictionary
    static const word propertiesName;


    // Constructors

        //- Construct from components
        turbulenceModel
        (
            const volVectorgpuField& U,
            const surfaceScalargpuField& alphaRhoPhi,
            const surfaceScalargpuField& phi,
            const word& propertiesName
        );


    //- Destructor
    virtual ~turbulenceModel() = default;


    // Member Functions

        //- Read model coefficients if they have changed
        virtual bool read() = 0;

        const Time& time() const
        {
            return runTime_;
        }

        const gpufvMesh& mesh() const
        {
            return mesh_;
        }

        //- Const access to the coefficients dictionary
        virtual const dictionary& coeffDict() const = 0;

        //- Helper function to return the name of the turbulence G field
        inline word GName() const
        {
            return word(type() + ":G");
        }

        //- Access function to velocity field
        inline const volVectorgpuField& U() const
        {
            return U_;
        }

        //- Access function to phase flux field
        inline const surfaceScalargpuField& alphaRhoPhi() const
        {
            return alphaRhoPhi_;
        }

        //- Return the volumetric flux field
        virtual tmp<surfaceScalargpuField> phi() const;

        //- Return the near wall distances
        const gpunearWallDist& y() const
        {
            return y_;
        }

        //- Return the laminar viscosity
        virtual tmp<volScalargpuField> nu() const = 0;

        //- Return the laminar viscosity on patch
        virtual tmp<scalargpuField> nu(const label patchi) const = 0;

        //- Return the turbulence viscosity
        virtual tmp<volScalargpuField> nut() const = 0;

        //- Return the turbulence viscosity on patch
        virtual tmp<scalargpuField> nut(const label patchi) const = 0;

        //- Return the effective viscosity
        virtual tmp<volScalargpuField> nuEff() const = 0;

        //- Return the effective viscosity on patch
        virtual tmp<scalargpuField> nuEff(const label patchi) const = 0;

        //- Return the laminar dynamic viscosity
        virtual tmp<volScalargpuField> mu() const = 0;

        //- Return the laminar dynamic viscosity on patch
        virtual tmp<scalargpuField> mu(const label patchi) const = 0;

        //- Return the turbulence dynamic viscosity
        virtual tmp<volScalargpuField> mut() const = 0;

        //- Return the turbulence dynamic viscosity on patch
        virtual tmp<scalargpuField> mut(const label patchi) const = 0;

        //- Return the effective dynamic viscosity
        virtual tmp<volScalargpuField> muEff() const = 0;

        //- Return the effective dynamic viscosity on patch
        virtual tmp<scalargpuField> muEff(const label patchi) const = 0;

        //- Return the turbulence kinetic energy
        virtual tmp<volScalargpuField> k() const = 0;

        //- Return the turbulence kinetic energy dissipation rate
        virtual tmp<volScalargpuField> epsilon() const = 0;

        //- Return the specific dissipation rate
        virtual tmp<volScalargpuField> omega() const = 0;

        //- Return the Reynolds stress tensor
        virtual tmp<volSymmTensorgpuField> R() const = 0;

        //- Validate the turbulence fields after construction
        //  Update derived fields as required
        virtual void validate();

        //- Solve the turbulence equations and correct the turbulence viscosity
        virtual void correct() = 0;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
