/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2019 OpenFOAM Foundation
    Copyright (C) 2017-2020 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "epsilonWallFunctionFvPatchScalargpuField.H"
#include "nutWallFunctionFvPatchScalargpuField.H"
#include "turbulenceModel.H"
#include "gpufvMatrix.H"
#include "addToRunTimeSelectionTable.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

const Foam::Enum
<
    Foam::epsilonWallFunctionFvPatchScalargpuField::blendingType
>
Foam::epsilonWallFunctionFvPatchScalargpuField::blendingTypeNames
({
    { blendingType::STEPWISE , "stepwise" },
    { blendingType::MAX , "max" },
    { blendingType::BINOMIAL , "binomial" },
    { blendingType::EXPONENTIAL, "exponential" }
});

Foam::scalar Foam::epsilonWallFunctionFvPatchScalargpuField::tolerance_ = 1e-5;


// * * * * * * * * * * * * * Protected Member Functions  * * * * * * * * * * //

namespace Foam
{
struct epsilonWallFunctionGraterThanToleranceFunctor : public std::unary_function<scalar,bool>{
	const scalar tolerance_;
	epsilonWallFunctionGraterThanToleranceFunctor(scalar tolerance): tolerance_(tolerance){}
    __host__ __device__
	bool operator () (const scalar& s){
		return s > tolerance_;
	}
};
}

void Foam::epsilonWallFunctionFvPatchScalargpuField::setMaster()
{
    if (master_ != -1)
    {
        return;
    }

    const volScalargpuField& epsilon =
        static_cast<const volScalargpuField&>(this->internalField());

    const volScalargpuField::Boundary& bf = epsilon.boundaryField();

    label master = -1;
    forAll(bf, patchi)
    {
        if (isA<epsilonWallFunctionFvPatchScalargpuField>(bf[patchi]))
        {
            epsilonWallFunctionFvPatchScalargpuField& epf = epsilonPatch(patchi);

            if (master == -1)
            {
                master = patchi;
            }

            epf.master() = master;
        }
    }
}

namespace Foam
{
struct epsilonWallFunctionFvPatchScalarFieldCreateWeightsFunctor: public std::unary_function<thrust::tuple<label,label>,label>
{
    __host__ __device__
    label operator()(const thrust::tuple<label,label>& t)
    {
        return thrust::get<1>(t) - thrust::get<0>(t);
    }
};
}


void Foam::epsilonWallFunctionFvPatchScalargpuField::createAveragingWeights()
{
    const volScalargpuField& epsilon =
        static_cast<const volScalargpuField&>(this->internalField());

    const volScalargpuField::Boundary& bf = epsilon.boundaryField();

    const gpufvMesh& mesh = epsilon.mesh();

    if (initialised_ && !mesh.hostmesh().changing())
    {
        return;
    }

    volScalargpuField weights
    (
        IOobject
        (
            "weights",
            mesh.time().timeName(),
            mesh.hostmesh(),
            IOobject::NO_READ,
            IOobject::NO_WRITE,
            false // do not register
        ),
        mesh,
        dimensionedScalar(dimless, Zero)
    );

    DynamicList<label> epsilonPatches(bf.size());
    forAll(bf, patchi)
    {
        if (isA<epsilonWallFunctionFvPatchScalargpuField>(bf[patchi]))
        {
            epsilonPatches.append(patchi);

            const labelgpuList& pcells = mesh.lduAddr().gpuPatchSortCells(patchi);
            const labelgpuList& losortStart = mesh.lduAddr().gpuPatchSortStartAddr(patchi);
            
            thrust::transform
            (
                thrust::make_permutation_iterator
                (
                    weights.begin(),
                    pcells.begin()
                ),
                thrust::make_permutation_iterator
                (
                    weights.begin(),
                    pcells.end()
                ),
                thrust::make_transform_iterator
                (
                    thrust::make_zip_iterator(thrust::make_tuple
                    (
                        losortStart.begin(),
                        losortStart.begin()+1
                    )),
                    epsilonWallFunctionFvPatchScalarFieldCreateWeightsFunctor()
                ),
                thrust::make_permutation_iterator
                (
                    weights.begin(),pcells.begin()
                ),
                addOperatorFunctor<scalar,label,scalar>()
            );
        }
    }

    cornerWeights_.setSize(bf.size());

    for (const auto& patchi : epsilonPatches)
    {
        const fvPatchScalargpuField& wf = weights.boundaryField()[patchi];
        cornerWeights_[patchi] = 1.0/wf.patchInternalField();
    }

    G_.setSize(internalField().size(), Zero);
    epsilon_.setSize(internalField().size(), Zero);

    initialised_ = true;
}


Foam::epsilonWallFunctionFvPatchScalargpuField&
Foam::epsilonWallFunctionFvPatchScalargpuField::epsilonPatch
(
    const label patchi
)
{
    const volScalargpuField& epsilon =
        static_cast<const volScalargpuField&>(this->internalField());

    const volScalargpuField::Boundary& bf = epsilon.boundaryField();

    const epsilonWallFunctionFvPatchScalargpuField& epf =
        refCast<const epsilonWallFunctionFvPatchScalargpuField>(bf[patchi]);

    return const_cast<epsilonWallFunctionFvPatchScalargpuField&>(epf);
}


void Foam::epsilonWallFunctionFvPatchScalargpuField::calculateTurbulenceFields
(
    const turbulenceModel& turbulence,
    scalargpuField& G0,
    scalargpuField& epsilon0
)
{
    // Accumulate all of the G and epsilon contributions
    forAll(cornerWeights_, patchi)
    {
        if (!cornerWeights_[patchi].empty())
        {
            epsilonWallFunctionFvPatchScalargpuField& epf = epsilonPatch(patchi);

            const gpuList<scalar>& w = cornerWeights_[patchi];

            epf.calculate(turbulence, w, epf.patch(), G0, epsilon0);
        }
    }

    // Apply zero-gradient condition for epsilon
    forAll(cornerWeights_, patchi)
    {
        if (!cornerWeights_[patchi].empty())
        {
            epsilonWallFunctionFvPatchScalargpuField& epf = epsilonPatch(patchi);

            epf == scalargpuField(epsilon0, epf.patch().gpuFaceCells());
        }
    }
}

namespace Foam
{
struct EpsilonCalculateEpsilonFunctor : public std::unary_function<label,scalar>
{
    const scalar Cmu75;
    const scalar kappa;
    const scalar Cmu25;
    scalar yPlus;
    const scalar yPlusLam;
    const scalar n;
    bool lowReCorrection;
    const label blend;
    const scalar* nuw;
    const scalar* cornerWeights;
    const scalar* y;
    const scalar* k;

    EpsilonCalculateEpsilonFunctor
    (
        const scalar Cmu75_,
        const scalar kappa_,
        const scalar Cmu25_,
        scalar yPlus_,
        const scalar yPlusLam_,
        const scalar n_,
        bool lowReCorrection_,
        const label blend_,
        const scalar* nuw_,
        const scalar* cornerWeights_,
        const scalar* y_,
        const scalar* k_
    ):
        Cmu75(Cmu75_),
        kappa(kappa_),
        Cmu25(Cmu25_),
        yPlus(yPlus_),
        yPlusLam(yPlusLam_),
        n(n_),
        lowReCorrection(lowReCorrection_),
        blend(blend_),
        nuw(nuw_),
        cornerWeights(cornerWeights_),
        y(y_),
        k(k_)
    {}

    __host__ __device__
    scalar operator()(const label& cellI,const label& faceI)
    {
        yPlus = Cmu25*y[faceI]*sqrt(k[cellI])/nuw[faceI];

        scalar w = cornerWeights[faceI];

        const scalar epsilonVis = w*2.0*k[cellI]*nuw[faceI]/sqr(y[faceI]);

        const scalar epsilonLog = w*Cmu75*pow(k[cellI], 1.5)/(kappa*y[faceI]);

        switch(blend)
        {
            case 0:
            {
                if (lowReCorrection && yPlus < yPlusLam)
                {
                    return epsilonVis;
                }
                else
                {
                    return epsilonLog;
                }
            }
                
            case 1:
            {
                return max(epsilonVis, epsilonLog);
            }

            case 2:
            {
                return pow
                (
                    pow(epsilonVis, n) + pow(epsilonLog, n),
                    1.0/n
                );
            }

            case 3:
            {
                const scalar Gamma = 0.001*pow4(yPlus)/(1.0 + yPlus);
                const scalar invGamma = 1.0/(Gamma + ROOTVSMALL);
                return epsilonVis*exp(-Gamma) + epsilonLog*exp(-invGamma);
            }
        }
    }
};

struct EpsilonCalculateGFunctor : public std::unary_function<label,scalar>
{
    const scalar Cmu25;
    const scalar kappa;
    scalar yPlus;
    const scalar yPlusLam;
    bool lowReCorrection;
    const scalar* cornerWeights;
    const scalar* y;
    const scalar* k;
    const scalar* nuw;
    const scalar* nutw;
    const scalar* magGradUw;

    EpsilonCalculateGFunctor
    (
        const scalar Cmu25_,
        const scalar kappa_,
        scalar yPlus_,
        const scalar yPlusLam_,
        bool lowReCorrection_,
        const scalar* cornerWeights_,
        const scalar* y_,
        const scalar* k_,
        const scalar* nuw_,
        const scalar* nutw_,
        const scalar* magGradUw_
    ):
        Cmu25(Cmu25_),
        kappa(kappa_),
        yPlus(yPlus_),
        yPlusLam(yPlusLam_),
        lowReCorrection(lowReCorrection_),
        cornerWeights(cornerWeights_),
        y(y_),
        k(k_),
        nuw(nuw_),
        nutw(nutw_),
        magGradUw(magGradUw_)
        {}

    __host__ __device__
    scalar operator()(const label& cellI,const label& faceI)
    {
        scalar w = cornerWeights[faceI];

        return
            w
           *(nutw[faceI] + nuw[faceI])
           *magGradUw[faceI]
           *Cmu25*sqrt(k[cellI])
           /(kappa*y[faceI]);
    }
};
}


void Foam::epsilonWallFunctionFvPatchScalargpuField::calculate
(
    const turbulenceModel& turbModel,
    const gpuList<scalar>& cornerWeights,
    const gpufvPatch& patch,
    scalargpuField& G0,
    scalargpuField& epsilon0
)
{
    const label patchi = patch.index();

    const nutWallFunctionFvPatchScalargpuField& nutw =
        nutWallFunctionFvPatchScalargpuField::nutw(turbModel, patchi);

    const scalargpuField& y = turbModel.y()[patchi];

    const tmp<scalargpuField> tnuw = turbModel.nu(patchi);
    const scalargpuField& nuw = tnuw();

    const tmp<volScalargpuField> tk = turbModel.k();
    const volScalargpuField& k = tk();

    const fvPatchVectorgpuField& Uw = turbModel.U().boundaryField()[patchi];

    const scalargpuField magGradUw(mag(Uw.snGrad()));

    const scalar Cmu25 = pow025(nutw.Cmu());
    const scalar Cmu75 = pow(nutw.Cmu(), 0.75);
    const scalar kappa = nutw.kappa();
    const scalar yPlusLam = nutw.yPlusLam();
    scalar yPlus = 0.0;
    label blending = 0;
   
    /*switch (blending_)
    {
        case blendingType::STEPWISE:
        {
            blending = 0;
            break;
        }

        case blendingType::MAX:
        {
            blending = 1;
            break;
        }

        case blendingType::BINOMIAL:
        {
            blending = 2;
            break;
        }

        case blendingType::EXPONENTIAL:
        {
            blending = 3;
            break;
        }
    }*/

    matrixPatchOperation
    (
        patchi,
        epsilon0,
        patch.boundaryMesh().mesh().lduAddr(),
        EpsilonCalculateEpsilonFunctor
        (
            Cmu75,
            kappa,
            Cmu25,
            yPlus,
            yPlusLam,
            n_,
            lowReCorrection_,
            blending,
            nuw.data(),
            cornerWeights.data(),
            y.data(),
            k.data()
        )
    );

    matrixPatchOperation
    (
        patchi,
        G0,
        patch.boundaryMesh().mesh().lduAddr(),
        EpsilonCalculateGFunctor
        (
            Cmu25,
            kappa,
            yPlus,
            yPlusLam,
            lowReCorrection_,
            cornerWeights.data(),
            y.data(),
            k.data(),
            nuw.data(),
            nutw.data(),
            magGradUw.data()
        )
    );
    // Set epsilon and G
/*    forAll(nutw, facei)
    {
        const label celli = patch.faceCells()[facei];

        const scalar yPlus = Cmu25*y[facei]*sqrt(k[celli])/nuw[facei];

        const scalar w = cornerWeights[facei];

        scalar epsilonBlended = 0.0;

        // Contribution from the viscous sublayer
        const scalar epsilonVis = w*2.0*k[celli]*nuw[facei]/sqr(y[facei]);

        // Contribution from the inertial sublayer
        const scalar epsilonLog =
            w*Cmu75*pow(k[celli], 1.5)/(nutw.kappa()*y[facei]);

        switch (blending_)
        {
            case blendingType::STEPWISE:
            {
                if (lowReCorrection_ && yPlus < nutw.yPlusLam())
                {
                    epsilonBlended = epsilonVis;
                }
                else
                {
                    epsilonBlended = epsilonLog;
                }
                break;
            }

            case blendingType::MAX:
            {
                // (PH:Eq. 27)
                epsilonBlended = max(epsilonVis, epsilonLog);
                break;
            }

            case blendingType::BINOMIAL:
            {
                // (ME:Eqs. 15-16)
                epsilonBlended =
                    pow
                    (
                        pow(epsilonVis, n_) + pow(epsilonLog, n_),
                        1.0/n_
                    );
                break;
            }

            case blendingType::EXPONENTIAL:
            {
                // (PH:p. 193)
                const scalar Gamma = 0.001*pow4(yPlus)/(1.0 + yPlus);
                const scalar invGamma = 1.0/(Gamma + ROOTVSMALL);
                epsilonBlended =
                    epsilonVis*exp(-Gamma) + epsilonLog*exp(-invGamma);
                break;
            }
        }

        epsilon0[celli] += epsilonBlended;

        if (!(lowReCorrection_ && yPlus < nutw.yPlusLam()))
        {
            G0[celli] +=
                w
               *(nutw[facei] + nuw[facei])
               *magGradUw[facei]
               *Cmu25*sqrt(k[celli])
               /(nutw.kappa()*y[facei]);
        }
    }*/
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::epsilonWallFunctionFvPatchScalargpuField::
epsilonWallFunctionFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    fixedValueFvPatchgpuField<scalar>(p, iF),
    blending_(blendingType::STEPWISE),
    n_(2.0),
    lowReCorrection_(false),
    initialised_(false),
    master_(-1),
    G_(),
    epsilon_(),
    cornerWeights_()
{}


Foam::epsilonWallFunctionFvPatchScalargpuField::
epsilonWallFunctionFvPatchScalargpuField
(
    const epsilonWallFunctionFvPatchScalargpuField& ptf,
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const fvPatchgpuFieldMapper& mapper
)
:
    fixedValueFvPatchgpuField<scalar>(ptf, p, iF, mapper),
    blending_(ptf.blending_),
    n_(ptf.n_),
    lowReCorrection_(ptf.lowReCorrection_),
    initialised_(false),
    master_(-1),
    G_(),
    epsilon_(),
    cornerWeights_()
{}


Foam::epsilonWallFunctionFvPatchScalargpuField::
epsilonWallFunctionFvPatchScalargpuField
(
    const gpufvPatch& p,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF,
    const dictionary& dict
)
:
    fixedValueFvPatchgpuField<scalar>(p, iF, dict),
    blending_
    (
        blendingTypeNames.getOrDefault
        (
            "blending",
            dict,
            blendingType::STEPWISE
        )
    ),
    n_
    (
        dict.getCheckOrDefault<scalar>
        (
            "n",
            2.0,
            scalarMinMax::ge(0)
        )
    ),
    lowReCorrection_(dict.getOrDefault("lowReCorrection", false)),
    initialised_(false),
    master_(-1),
    G_(),
    epsilon_(),
    cornerWeights_()
{
    // Apply zero-gradient condition on start-up
    this->operator==(patchInternalField());
}


Foam::epsilonWallFunctionFvPatchScalargpuField::
epsilonWallFunctionFvPatchScalargpuField
(
    const epsilonWallFunctionFvPatchScalargpuField& ewfpsf
)
:
    fixedValueFvPatchgpuField<scalar>(ewfpsf),
    blending_(ewfpsf.blending_),
    n_(ewfpsf.n_),
    lowReCorrection_(ewfpsf.lowReCorrection_),
    initialised_(false),
    master_(-1),
    G_(),
    epsilon_(),
    cornerWeights_()
{}


Foam::epsilonWallFunctionFvPatchScalargpuField::
epsilonWallFunctionFvPatchScalargpuField
(
    const epsilonWallFunctionFvPatchScalargpuField& ewfpsf,
    const DimensionedgpuField<scalar, gpuvolMesh>& iF
)
:
    fixedValueFvPatchgpuField<scalar>(ewfpsf, iF),
    blending_(ewfpsf.blending_),
    n_(ewfpsf.n_),
    lowReCorrection_(ewfpsf.lowReCorrection_),
    initialised_(false),
    master_(-1),
    G_(),
    epsilon_(),
    cornerWeights_()
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

Foam::scalargpuField& Foam::epsilonWallFunctionFvPatchScalargpuField::G
(
    bool init
)
{
    if (patch().index() == master_)
    {
        if (init)
        {
            G_ = 0.0;
        }

        return G_;
    }

    return epsilonPatch(master_).G();
}


Foam::scalargpuField& Foam::epsilonWallFunctionFvPatchScalargpuField::epsilon
(
    bool init
)
{
    if (patch().index() == master_)
    {
        if (init)
        {
            epsilon_ = 0.0;
        }

        return epsilon_;
    }

    return epsilonPatch(master_).epsilon(init);
}


void Foam::epsilonWallFunctionFvPatchScalargpuField::updateCoeffs()
{
    if (updated())
    {
        return;
    }

    const turbulenceModel& turbModel = db().lookupObject<turbulenceModel>
    (
        IOobject::groupName
        (
            turbulenceModel::propertiesName,
            internalField().group()
        )
    );

    setMaster();

    if (patch().index() == master_)
    {
        createAveragingWeights();
        calculateTurbulenceFields(turbModel, G(true), epsilon(true));
    }

    const scalargpuField& G0 = this->G();
    const scalargpuField& epsilon0 = this->epsilon();

    typedef DimensionedgpuField<scalar, gpuvolMesh> FieldType;

    FieldType& G = db().lookupObjectRef<FieldType>(turbModel.GName());

    FieldType& epsilon = const_cast<FieldType&>(internalField());

    thrust::copy(thrust::make_permutation_iterator(G0.begin(),patch().gpuFaceCells().begin()),
                 thrust::make_permutation_iterator(G0.begin(),patch().gpuFaceCells().end()),
                 thrust::make_permutation_iterator(G.begin(),patch().gpuFaceCells().begin()));
                 
    thrust::copy(thrust::make_permutation_iterator(epsilon0.begin(),patch().gpuFaceCells().begin()),
                 thrust::make_permutation_iterator(epsilon0.begin(),patch().gpuFaceCells().end()),
                 thrust::make_permutation_iterator(epsilon.begin(),patch().gpuFaceCells().begin()));

    fvPatchgpuField<scalar>::updateCoeffs();
}

namespace Foam
{
struct epsilonWallFunctionFvPatchScalarFieldupdateCoeffsFunctor
{
	const scalar tolerance_;
	epsilonWallFunctionFvPatchScalarFieldupdateCoeffsFunctor(scalar tolerance):
		tolerance_(tolerance){}
    __host__ __device__
    scalar operator () (const scalar& w,const thrust::tuple<scalar,scalar>& t)
    {
        scalar G = thrust::get<0>(t);
        scalar G0 = thrust::get<1>(t);
		
        if(w>tolerance_)return (1.0-w)*G+w*G0;
    }	
	
};
}

void Foam::epsilonWallFunctionFvPatchScalargpuField::updateWeightedCoeffs
(
    const scalargpuField& weights
)
{
    if (updated())
    {
        return;
    }

    const turbulenceModel& turbModel = db().lookupObject<turbulenceModel>
    (
        IOobject::groupName
        (
            turbulenceModel::propertiesName,
            internalField().group()
        )
    );

    setMaster();

    if (patch().index() == master_)
    {
        createAveragingWeights();
        calculateTurbulenceFields(turbModel, G(true), epsilon(true));
    }

    const scalargpuField& G0 = this->G();
    const scalargpuField& epsilon0 = this->epsilon();

    typedef DimensionedgpuField<scalar, gpuvolMesh> FieldType;

    FieldType& G = db().lookupObjectRef<FieldType>(turbModel.GName());

    FieldType& epsilon = const_cast<FieldType&>(internalField());

    scalargpuField& epsilonf = *this;

    // Only set the values if the weights are > tolerance
/*    forAll(weights, facei)
    {
        const scalar w = weights[facei];

        if (w > tolerance_)
        {
            const label celli = patch().faceCells()[facei];

            G[celli] = (1.0 - w)*G[celli] + w*G0[celli];
            epsilon[celli] = (1.0 - w)*epsilon[celli] + w*epsilon0[celli];
            epsilonf[facei] = epsilon[celli];
        }
    }
    */
    thrust::transform
    (
        weights.begin(),
        weights.end(),
        thrust::make_zip_iterator(thrust::make_tuple
        (
            thrust::make_permutation_iterator(G.begin(),  patch().gpuFaceCells().begin()),
            thrust::make_permutation_iterator(G0.begin(),  patch().gpuFaceCells().begin())
        )),
        thrust::make_permutation_iterator(G.begin(),patch().gpuFaceCells().begin()),
        epsilonWallFunctionFvPatchScalarFieldupdateCoeffsFunctor(tolerance_)
    );
                     
    thrust::transform
    (
        weights.begin(),
        weights.end(),
        thrust::make_zip_iterator(thrust::make_tuple
        (
            thrust::make_permutation_iterator(epsilon.begin(),  patch().gpuFaceCells().begin()),
            thrust::make_permutation_iterator(epsilon0.begin(),  patch().gpuFaceCells().begin())
        )),
        thrust::make_permutation_iterator(epsilon.begin(),patch().gpuFaceCells().begin()),
        epsilonWallFunctionFvPatchScalarFieldupdateCoeffsFunctor(tolerance_)
    );
                     
    thrust::copy
    (
        thrust::make_permutation_iterator(epsilon.begin(),patch().gpuFaceCells().begin()),
        thrust::make_permutation_iterator(epsilon.begin(),patch().gpuFaceCells().end()),
        epsilonf.begin()
    );

    fvPatchgpuField<scalar>::updateCoeffs();
}


void Foam::epsilonWallFunctionFvPatchScalargpuField::manipulateMatrix
(
    gpufvMatrix<scalar>& matrix
)
{
    if (manipulatedMatrix())
    {
        return;
    }

    matrix.setValues(patch().gpuFaceCells(), patchInternalField());

    fvPatchgpuField<scalar>::manipulateMatrix(matrix);
}


void Foam::epsilonWallFunctionFvPatchScalargpuField::manipulateMatrix
(
    gpufvMatrix<scalar>& matrix,
    const gpuField<scalar>& weights
)
{
    if (manipulatedMatrix())
    {
        return;
    }

    labelgpuList constraintCells(weights.size());
    scalargpuList constraintValues(weights.size());
    const labelgpuList& faceCells = patch().gpuFaceCells();

    const DimensionedgpuField<scalar, gpuvolMesh>& fld = internalField();

    typename gpuList<label>::iterator end = 
        thrust::copy_if
        (
            faceCells.begin(),
            faceCells.end(),
            weights.begin(),
            constraintCells.begin(),
            epsilonWallFunctionGraterThanToleranceFunctor(tolerance_)
        );
                               
    label nConstrainedCells = end - constraintCells.begin();
    
    constraintCells.setSize(nConstrainedCells);
    
    thrust::copy_if
    (
        thrust::make_permutation_iterator(fld.begin(),faceCells.begin()),
        thrust::make_permutation_iterator(fld.begin(),faceCells.end()),
        weights.begin(),
        constraintValues.begin(),
        epsilonWallFunctionGraterThanToleranceFunctor(tolerance_)
    );
                               
    constraintValues.setSize(nConstrainedCells);

    if (debug)
    {
        Pout<< "Patch: " << patch().name()
            << ": number of constrained cells = " << constraintCells.size()
            << " out of " << patch().size()
            << endl;
    }

    thrust::sort_by_key
    (
        constraintCells.begin(),
        constraintCells.end(),
        constraintValues.begin()
    );

    matrix.setValues(constraintCells, constraintValues);

    fvPatchgpuField<scalar>::manipulateMatrix(matrix);
}


void Foam::epsilonWallFunctionFvPatchScalargpuField::write
(
    Ostream& os
) const
{
    os.writeEntry("lowReCorrection", lowReCorrection_);
    os.writeEntry("blending", blendingTypeNames[blending_]);
    os.writeEntry("n", n_);
    fixedValueFvPatchgpuField<scalar>::write(os);
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
    makePatchTypeField
    (
        fvPatchScalargpuField,
        epsilonWallFunctionFvPatchScalargpuField
    );
}


// ************************************************************************* //
