/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

inline Foam::ArrheniusReactionRate::ArrheniusReactionRate
(
    const scalar A,
    const scalar beta,
    const scalar Ta
)
:
    A_(A),
    beta_(beta),
    Ta_(Ta),
    gA_(nullptr),
    gbeta_(nullptr),
    gTa_(nullptr)
{
	hipMalloc(&gA_, sizeof(scalar));
	hipMalloc(&gbeta_, sizeof(scalar));
	hipMalloc(&gTa_, sizeof(scalar));
	hipMemcpy(gA_, &A_, sizeof(scalar), hipMemcpyHostToDevice);
	hipMemcpy(gbeta_, &beta_, sizeof(scalar), hipMemcpyHostToDevice);
	hipMemcpy(gTa_, &Ta_, sizeof(scalar), hipMemcpyHostToDevice);
}


inline Foam::ArrheniusReactionRate::ArrheniusReactionRate
(
    const speciesTable&,
    const dictionary& dict
)
:
    A_(dict.get<scalar>("A")),
    beta_(dict.get<scalar>("beta")),
    Ta_(dict.get<scalar>("Ta")),
	gA_(nullptr),
	gbeta_(nullptr),
	gTa_(nullptr)

{
	hipMalloc(&gA_, sizeof(scalar));
	hipMalloc(&gbeta_, sizeof(scalar));
	hipMalloc(&gTa_, sizeof(scalar));
	hipMemcpy(gA_, &A_, sizeof(scalar), hipMemcpyHostToDevice);
	hipMemcpy(gbeta_, &beta_, sizeof(scalar), hipMemcpyHostToDevice);
	hipMemcpy(gTa_,&Ta_, sizeof(scalar), hipMemcpyHostToDevice);
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

__host__ __device__
inline Foam::scalar Foam::ArrheniusReactionRate::operator()
(
    const scalar p,
    const scalar T,
    const scalarField&
) const
{
    scalar ak = *gA_;

    if (mag(*gbeta_) > VSMALL)
    {
        ak *= pow(T, *gbeta_);
    }

    if (mag(*gTa_) > VSMALL)
    {
        ak *= exp(-*gTa_/T);
    }

    return ak;
}


inline void Foam::ArrheniusReactionRate::write(Ostream& os) const
{
    os.writeEntry("A", A_);
    os.writeEntry("beta", beta_);
    os.writeEntry("Ta", Ta_);
}


inline Foam::Ostream& Foam::operator<<
(
    Ostream& os,
    const ArrheniusReactionRate& arr
)
{
    arr.write(os);
    return os;
}


// ************************************************************************* //
